#!/bin/sh -p

#################################################################################
# Copyright (c) 2013-2025, SAS Institute Inc., Cary, NC, USA. All Rights Reserved.
#################################################################################

LANG=C; export LANG

cmd=laxcheck
ver='SAS 9.4'

minoracle=8.0
minredhat=8.0
minsuse=15.0
minsles=15.0

min_glibcVersion=02.11.01
min_bitness=64bit

problem=no

emit() {
   echo $cmd: "$@" >&2
}

# this is for print out current and minimum version information
stdoutmsg() {
   echo "$@" >&1
}

un=`uname`
case "$un" in
Linux*);;
*)   emit "The uname command reports \"$un\", not \"Linux\""
     emit "Is this a Linux system?"
     problem=yes
     exit 1;;
esac

check64=`uname -a|grep x86_64`

if [ x"$check64" = x ]
then
     bitness=""
     problem=yes
else
    bitness=64bit
fi

if [ ! -e /bin/ksh ]
then
     emit "WARNING /bin/ksh must be installed if the SAS OS Scheduler will be used. It is not installed currently."
     ##problem=no
elif /bin/ksh -c 'exit 0'
then
     : okay
else
     emit "/bin/ksh exists but does not appear to work properly"
     ##problem=no
fi


####################################################
# This determines which release the machine is
####################################################

os=""

if [ -f /etc/oracle-release ] ; then
	os=`cat /etc/oracle-release`
	oracle=`echo $os|grep Oracle`
	oracle=${oracle##*release }
	oracle=${oracle%% *}
elif [ -f /etc/redhat-release ] ; then

	os=`cat /etc/redhat-release`
	redhat=`echo $os|grep Red`
	redhat=${redhat##*release }
	redhat=${redhat%% *}
elif [ -f /etc/SuSE-release ] ; then
	os=`cat /etc/SuSE-release`
	# match on content with "PATCHLEVEL"
	suse=`echo $os | sed -n 's/.*SUSE.*VERSION = \([0-9.]*\).*PATCHLEVEL = \([0-9]*\).*/\1.\2/p'`
	#match on content without "PATCHLEVEL if no match above"	
	if [ x"$suse" == x ]  ; then
		suse=`echo $os | sed -n 's/.*SUSE.*VERSION = \([0-9.]*\).*/\1/p'`
	fi	
elif [ -f /etc/os-release ] ; then
	os=`cat /etc/os-release`
	# match on content with "PATCHLEVEL"
	sles=`echo $os | sed -n 's/.*VERSION_ID="\([0-9.]*\)".*/\1/p'`
	#match on content without "PATCHLEVEL if no match above"	
	if [ x"$suse" == x ]  ; then
		suse=`echo $os | sed -n 's/.*SUSE.*VERSION = \([0-9.]*\).*/\1/p'`
	fi	
else 
	for each in `ls /etc/*release`; do
		if [ -f "$each" ] ; then
			#assume that it's a different distribution of redhat with Red in the text.
			os=`cat $each` 
			redhat=`cat $each | grep Red`
			redhat=${redhat##*release }
			redhat=${redhat%% *}
			if [ x"$redhat" != x ] ; then
				break;
			fi
		fi
	done
fi

if [ x"$oracle" != x ] ; then
   if awk "BEGIN {exit !("$oracle" < "$minoracle")}" ; then
      emit "Your Oracle Linux version $oracle is outdated, $ver requires $minoracle"
      problem=yes
   fi
elif [ x"$redhat" != x ] ; then
   if awk "BEGIN {exit !("$redhat" < "$minredhat")}" ; then
      emit "Your Red Hat version $redhat is outdated, $ver requires $minredhat"
      problem=yes
   fi
elif [ x"$suse" != x ] ; then
   if awk "BEGIN {exit !("$suse" < "$minsuse")}" ; then
      emit "Your SUSE version $suse is outdated, $ver requires $minsuse"
      problem=yes
   fi
elif [ x"$sles" != x ] ; then
   if awk "BEGIN {exit !("$sles" < "$minsles")}" ; then
      emit "Your SLES version $sles is outdated, $ver requires $minsles"
      problem=yes
   fi
   ## case where alternate OS might exist such as CentOS
else emit "WARNING You are using an alternate operating system. See section SAS Support for Alternative Operating Systems at http://support.sas.com/techsup/pcn/altopsys.html for support statement."
fi

glibcVersion=`ldd --version | sed q`
glibcVersion=${glibcVersion##* }
unmodified="$glibcVersion"

glibcVersion=`echo "$glibcVersion" | sed -e 's/^\([0-9]\)\./0\1./' -e 's/\.\([0-9]\)\./.0\1./g' -e 's/\.\([0-9]\)$/.0\1/'`
glibcVersion=`echo "$glibcVersion" | sed -e 's/^\([0-9]\)\./0\1./' -e 's/\.\([0-9]\)\./.0\1./g' -e 's/\.\([0-9]\)$/.0\1/'`
if awk "BEGIN {exit !("$glibcVersion" < "$min_glibcVersion")}"
then
   emit "$ver Foundation requires GLIBC $min_glibcVersion"
   emit "The GLIBC version detected was $unmodified"
   problem=yes
fi


################################################################
# This is the message for SDW to pharse the platform and version
################################################################

if [ x"$oracle" != x ]  ; then
   stdoutmsg "Oracle $oracle $bitness GLIBC $glibcVersion"
   stdoutmsg "Oracle $minoracle $min_bitness GLIBC $min_glibcVersion"
elif [ x"$redhat" != x ] ; then
   stdoutmsg "RedHat $redhat $bitness GLIBC $glibcVersion"
   stdoutmsg "RedHat $minredhat $min_bitness GLIBC $min_glibcVersion"
elif [ x"$suse" != x ]  ; then
   stdoutmsg "SUSE $suse $bitness GLIBC $glibcVersion"
   stdoutmsg "SUSE $minsuse $min_bitness GLIBC $min_glibcVersion"
elif [ x"$sles" != x ]  ; then
   stdoutmsg "SLES $sles $bitness GLIBC $glibcVersion"
   stdoutmsg "SLES $minsles $min_bitness GLIBC $min_glibcVersion"
else
   stdoutmsg "OS $os"
   stdoutmsg "Linux OS $min_bitness GLIBC $min_glibcVersion"
  ##problem=yes
fi


case $problem in
no)  emit "Checker for $ver finished, with no detected problems"
     exit 0;;
yes) emit "Checker for $ver finished, with problems"
     exit 1;;
esac
