#!/bin/sh
#
#
# Change dirs to 0755 (everybody read/search, owner read/write/search)
#

################################################################################
# Copyright (c) 2013 - 2025, SAS Institute Inc., Cary, NC, USA. All Rights Reserved.
################################################################################

targetdir="<INSTALL_DIR>"

#
# Function to log error messages, which sends the date and the error message
# to stderr
#
log () {
   date >&2
   echo "$@" >&2 
}

#
# Make sure the directory path has been set, and is not the root directory
#
case "$targetdir" in
   "") log "ERROR: Directory name set to nothing"; exit 1;;
   "/") log "ERROR: Starting from Root directory not allowed"; exit 1;;
esac

#
# Does the directory exist and is it really a directory
#
if [ ! -d "$targetdir" ] ; then
   log "ERROR: $targetdir is not a directory"; exit 1
fi

#
# Change directory to SASHOME
#
cd "$targetdir" || { 
   log "ERROR: Can not change directory to $targetdir"; exit 1
}

#
# Create a temporary file so that xargs always has a file to work on, or 
# otherwise it could through an error on some platforms
#
tmpfilename=".tmpfile$$"
touch "$tmpfilename" || {
   log "ERROR: Could not create temporary file: ${targetdir}/${tmpfilename}"
}

#
# Step 1
#
# Change normal files to 0644 (everybody read, owner read/write)
#

find . \( -type f -o -name _jvm -prune \) | 
   sed -e '/^.$/d' -e '/^..$/d' -e "s,.,\'&," -e "s,\$,\'," |
   xargs chmod 0644 "$tmpfilename"


#
# Change dirs to 0755 (everybody read/search, owner read/write/search)
# Step 2
#

find . -type d | 
   sed -e '/^.$/d' -e '/^..$/d' -e "s,.,\'&," -e "s,\$,\'," |
   xargs chmod 0755 "$tmpfilename"

#
#Step 3
#
# Change Bourne shell scripts to executable (owner read/write/execute,
# everyone read/execute)
#

find . -type f -name \*\.sh | 
   sed -e '/^.$/d' -e '/^..$/d' -e "s,.,\'&," -e "s,\$,\'," |
   xargs chmod 0755 "$tmpfilename"

#
# Check files for the "#!/bin/sh" tag to see if they are scripts.
# Ignore files ending in .jar, and anything in a doc directory
#

#find . \( -type f -a \( \! -name \*.jar -o -name doc -prune \) \) \
#-exec egrep -s "\#\!/bin/sh" \{\} \; -exec chmod 0755 \{\} \;

#
#Step 4
#
# Change REXX execs to executable for z/OS (owner read/write/execute,
# everyone read/execute)
#

find . -type f -name \*\.rexx | 
   sed -e '/^.$/d' -e '/^..$/d' -e "s,.,\'&," -e "s,\$,\'," |
   xargs chmod 0755 "$tmpfilename"

# Check for binary executable uninstallers, which will match the
# pattern "Uninst*"
#

find . -type f -name Uninst\* | 
   sed -e '/^.$/d' -e '/^..$/d' -e "s,.,\'&," -e "s,\$,\'," |
   xargs chmod 0755 "$tmpfilename"

# Check for executable files by checking magic cookies
find . \( -type f -a \( \! -name \*.jar -o -name doc -prune \) \) -print |
while read ln
do
   set -- `dd if="$ln" count=1 bs=2 2> /dev/null | od -x | head -n1`
   bytes=${2:-NULL}
   if [ $bytes != NULL  -a \
        $bytes = "feca" -o \
        $bytes = "cafe" -o \
        $bytes = "2123" -o \
        $bytes = "2321" -o \
        $bytes = "0210" -o \
        $bytes = "0214" -o \
        $bytes = "020b" -o \
        $bytes = "7f45" -o \
        $bytes = "01df" -o \
        $bytes = "01f7" -o \
        $bytes = "457f" -o \
        $bytes = "0183" ] ; then

      chmod a+rx "$ln"
   fi
done

rm "$tmpfilename" || {
   log "ERROR: Could not delete temporary file: ${targetdir}/${tmpfilename}"
}

exit 0


#
# Done
#
