/* Rexx                                                                      */
/*****************************************************************************/
/*  PRODUCT: MVS SAS                                                         */
/*  PURPOSE: Tapedump process to replace values in USS                       */
/*  FROM: SAS INSTITUTE INC., SAS Campus Drive, Cary, NC 27513               */
/*****************************************************************************/
/* Changed 06Dec2010 RVC - Check RC after './repluss' path','string','c.tag  */
/* Changed 31JAN2011 GMK - Added strip(string) line in add_repl_file         */
/* Changed 15Mar2011 RVC - Corrected parsing of ORIGINAL_xxx in read_cust:   */
/* Changed 16Mar2011 RVC - Write all buffers not just with string found      */
/* Changed 15Jun2011 RVC - Exclude files that shouldn't be changed           */
/* Changed 16Jun2011 GMK - changed | to & in compound cond for excl+ignore   */
/* Changed 17Jun2011 RVC - Changed filename exclude to dir.i exclude         */
/*****************************************************************************/
/* Find a String in a USS Filesystem and Optionally Replace it               */
/* Format:                                                                   */
/*  repluss <path | path/filename>,search_string                             */
/*          ,<replace_string | SASGENERATE=<SAS_INIT | REPLACEMENTS | tag> > */
/*  Where:                                                                   */
/*   path           = Path - ie /usr/lpp/SAS93                               */
/*   path/filename  = File - ie /usr/lpp/SAS93/deploymntreg/registry.xml     */
/*   search_string  = String to search for (case sensitive)                  */
/*   replace_string = Relacement for search_string (case sensitive)          */
/*   SASGENERATE=   = Keyword for Generating of Control Files                */
/*   SAS_INIT       = Keyword for Initialization of Control Files            */
/*   REPLACEMENTS   = Keyword for Changing strings given in Control Files    */
/*   tag            = Keyword for Tagging information in Control Files       */
/* Format for initializing file:                                             */
/*  repluss path,dummy,SASGENERATE=SAS_INIT                                  */
/*  Example: /tmp/SAS92P/jobs,dummy,SASGENERATE=SAS_INIT                     */
/* Format for creating customer modifyable tags:                             */
/*  repluss <path | path/filename>,search_string,SASGENERATE=tag             */
/*  Example: repluss /usr/lpp/SAS93,TDI.SAS93.,SASGENERATE=HLQ               */
/* Format for replacing strings:                                             */
/*  repluss <path | path/filename>,search_string,replace_string              */
/*  Example: repluss /usr/lpp/SAS93,TDI.SAS93.,ABC.SAS93.                    */
/* Format for replacing strings using the Control Files                      */
/*  repluss path,dummy,SASGENERATE=REPLACEMENTS                              */
/*  Example: /tmp/SAS92P/jobs,dummy,SASGENERATE=REPLACEMENTS                 */
/* Notes:                                                                    */
/*  If file rplsasin is found at SASGENERATE=SAS_INIT then repluss will      */
/*  use the file's tag and search_string to generate the control files       */
/*  instead of having to run repluss SASGENERATE=tag separately for each tag */
/*  and search_string.                                                       */
/*****************************************************************************/
/*trace i*/
/*****************************************************************************/
/* Set up for System Calls & who we are                                      */
/*****************************************************************************/
parse source system . program .                   /* Assign the program name */
program = substr(program,lastpos('/',program)+1)           /* Strip off path */
program = substr(program,lastpos('\',program)+1)           /* Strip off path */
if system ^= 'TSO' then do                               /* Must run on z/OS */
   say program': Cannot be Run From' system                 /* Write message */
   exit 16                                               /* Return to caller */
  end
call syscalls 'ON'                                     /* Allow System calls */
address 'SYSCALL'                    /* Default the commands to System calls */

/*****************************************************************************/
/* Main Process - call major subroutines to do processing                    */
/*****************************************************************************/
parse arg path '!' string '!' out    /* Get path, search string, replacement */
out = strip(out)                                   /* Remove trailing blanks */
call initialize                                /* Initialize variables, etc. */
if translate(substr(out,1,12)) = 'SASGENERATE=' then do   /* Generate wanted */
   if length(out) < 13 then,                                  /* Missing tag */
      call exit '16~SASGENERATE Tag Missing'
   generate = translate(substr(out,13))                           /* Get tag */
   tag = generate string';'                        /* Set the tag for search */
   out = ''                                               /* Force no output */
   select
    when generate = 'SAS_INIT' then do        /* Request to initialize Files */
      'chdir' path                                         /* Go to the path */
      if rc ^= 0 | errno ^= 0 | errnojr ^= 0 | retval = -1 then, /* Check RC */
         call exit rc'~Error Finding Directory:' path
      call read_cust 'init'                /* Read the customer control file */
      call init_files                         /* Initialize the output files */
      call read_sas                             /* Read the SAS control file */
     end
    when generate = 'REPLACEMENTS' then do     /* Request to replace strings */
      'chdir' path                                         /* Go to the path */
      if rc ^= 0 | errno ^= 0 | errnojr ^= 0 | retval = -1 then, /* Check RC */
         call exit rc'~Error Finding Directory:' path
      call read_cust                       /* Read the Customer Control file */
      call do_replace                               /* Do a full replacement */
      call exit '0~Execution Successful'                       /* Go get out */
     end
    otherwise out = ''                       /* Blank out the replace string */
   end
  end
 else do                                     /* We may be replacing someting */
   if pos(string,out) ^= 0 then,      /* Search string in replacement string */
   call exit '16~Search String ('string') Within Replacement String('out').',
             'This Would Cause an Endless Loop.'                    /* Error */
   call read_cust                          /* Read the Customer Control File */
   if path <= length(orig_saspath) then,         /* Prepend customer to path */
      path = saspath||substr(path,length(orig_saspath)+1) /* Prepend to path */
    else path = saspath                                 /* Just replace path */
  end
if tag = '' then tag = 'sas_single_search' string';'  /* Set customer search */
next = 0                                             /* Initialize a counter */
call get_files path         /* Get a list of the files in the path specified */
files.0 = next                                        /* Set the array count */
say ''                                                              /* Write */
if files.0 = 0 then do                              /* Single file specified */
   files.1 = path                                  /* Assign the single file */
   files.0 = 1                                        /* Set the array count */
  end
 else say files.0 'Searchable Files Found'                 /* Multiple files */
call read_files                                     /* Read each file listed */
call exit '0~Execution Successful'         /* Return to caller - all is well */

/*****************************************************************************/
/* Get a list of the files to be searched based on the path specified        */
/*****************************************************************************/
get_files: procedure expose files. ignore exclude next in_process    /* Vars */
 parse arg path .                                 /* Get the [sub]-directory */
 if pos('/help/',path) ^= 0 then return            /* Don't check help files */
 'lstat' path 'filest.'      /* Check the path to see if it is really a file */
 if rc ^= 0 | errno ^= 0 | errnojr ^= 0 | retval = -1 then,  /* Check the RC */
    call exit '0~Error Extracting Information About:' path
 if filest.1 = 2 | filest.1 = 3 then return  /* The path specified is a file */
 if substr(path,length(path),1) ^= '/' then path = path'/'         /* Append */
 'readdir' path 'dir. dirst.'                          /* Read the directory */
 if rc ^= 0 | errno ^= 0 | errnojr ^= 0 | retval = -1 then,  /* Check the RC */
    call exit rc'~Error Listing Path:' path
 do i = 3 to dir.0                                /* Read the directory list */
  select
   when dirst.i.1 = 1 then call get_files path||dir.i  /* Is a sub-directory */
   when dirst.i.1 = 5 then nop                     /* Symbolic link - ignore */
   otherwise do                                            /* This is a file */
     if lastpos('.',dir.i) ^= 0 then,                /* Is there a filetype? */
        filetype = substr(dir.i,lastpos('.',dir.i)+1)        /* Get filetype */
      else filetype = 'WeShouldNeverFindThisFileTypeInSAS'
     if wordpos(filetype,ignore) = 0 & wordpos(dir.i,exclude) = 0 then do
        next = next + 1                             /* Increment the counter */
        files.next = path||dir.i                        /* Add to file array */
       end
    end
  end
 end
return

/*****************************************************************************/
/* Search for the input string and replace it with the output string         */
/*****************************************************************************/
read_files:
 do f = 1 to files.0                      /* All files in the path specified */
  file = files.f                        /* Get the path and name of the file */
  do i = 1 to words(codeset)                     /* Search in EBCDIC & ascii */
   found = 0                            /* Initialize string found indicator */
   work_tag = tag                                /* Put tag in work variable */
   call open_file                                              /* Open files */
   if rc ^= 0 | retval = -1 then iterate f              /* No sense going on */
   call f_control_file         /* Issue f_control commands to translate file */
   if rc ^= 0 | retval = -1 then iterate f              /* No sense going on */
   do j = 1 by 1 until work_tag = ''           /* Search for all the strings */
    parse var work_tag generate string ';' work_tag    /* Get tag and string */
    if generate = 'sas_single_search' then generate = ''  /* Customer search */
    if j = 1 then say 'Searching ('word(codeset,i)'):' file       /* Write */
    call read_file                                          /* Read the File */
   end
   call close_file                                         /* Close the file */
  end
 end
return                                                   /* Return to caller */

/*****************************************************************************/
/* Open files                                                                */
/*****************************************************************************/
open_file:
 'open' file O_RDONLY '000'                           /* Open the input file */
 if rc ^= 0 | retval = -1 then do                        /* Check for errors */
    say 'Error Opening Input File. RC='errno', Reason Code:' errnojr
    say 'File='file                                                 /* Write */
    set_rc = 4                                  /* Set a warning return code */
    return                                               /* Return to caller */
   end
  else in_fd = retval                    /* Assign the input file descriptor */
 size = stream(file,'C','query size')            /* Get the size of the file */
 if out ^= '' then do                               /* Replacement specified */
    'open' file'.updated' O_CREAT+O_TRUNC+O_WRONLY '755'        /* Open file */
    if rc ^= 0 | retval = -1 then do                     /* Check for errors */
       say 'Error Opening Output File. RC='errno', Reason Code:' errnojr
       call exit rc'~File='file'.updated'                   /* Report errors */
      end
     else out_fd = retval               /* Assign the output file descriptor */
   end
return                                                   /* Return to caller */

/*****************************************************************************/
/* Issue f_control commands to translate the file when read later on         */
/*****************************************************************************/
f_control_file:
 if word(codeset,i) = 'ascii' then do           /* Do translation  for ascii */
    pccsid = '0417'x          /* EBCDIC Program character codeset identifier */
    fccsid = '0333'x           /* ascii Program character codeset identifier */
    'f_control_cvt (in_fd) cvt_setcvton pccsid fccsid'          /* Translate */
    if rc ^= 0 | retval = -1 then do                     /* Check for errors */
       say 'Error Setting Translation on For Input File.',          /* Write */
           'RC='errno', Reason Code:' errnojr
       say 'File='file                                      /* Report errors */
       set_rc = 4                               /* Set a warning return code */
       return                                            /* Return to caller */
      end
    if out ^= '' then do                            /* Replacement specified */
       'f_control_cvt (out_fd) cvt_setcvton pccsid fccsid'     /* Translate */
       if rc ^= 0 | retval = -1 then do                 /* Check for errors */
          say 'Error Setting Translation on For Output File.',
              'RC='errno', Reason Code:' errnojr                   /* Write */
          call exit rc'~File='file'.updated'               /* Report errors */
         end
      end
   end
return                                                   /* Return to caller */

/*****************************************************************************/
/* Read the target file                                                      */
/*****************************************************************************/
read_file:
 in_process = 0                                /* Initialize in_process flag */
 read_size = 0       /* Initialize the amount of the file that has been read */
 'lseek' in_fd 0 seek_set        /* Start reading from the beginning of file */
 do s = 1 by 1 until read_size >= size  /* Loop until file has been all read */
  'read' in_fd 'line' 16384                      /* Read in a buffer of data */
  if rc ^= 0 | retval = -1 then do                       /* Check for errors */
     say 'Error Reading Input File. RC='errno', Reason Code:' errnojr
     say 'File='file                                        /* Report errors */
     set_rc = 4                                 /* Set a warning return code */
     return                                              /* Return to caller */
    end
  buffer_size = length(line)           /* Get the actual size of buffer read */
  line_out = ''                              /* Initialize the output buffer */
  select
   when maybe then start = 1             /* In case the string spans buffers */
   when pos(string,line) ^= 0 then do                    /* String in buffer */
     work = line                             /* Copy line to a work location */
     position = 0                   /* Initialize position of string in work */
     roffset = 1                     /* Initialize search offset into buffer */
     offset = 1                        /* Initialize search offset into work */
     do until pos(string,work,offset) = 0                             /* All */
      rposition = pos(string,line,roffset)                  /* Real position */
      position = pos(string,work,offset)                 /* Position in work */
      if generate ^= '' then do                        /* Generate specified */
         call add_repl_file         /* Add entry to replacement control file */
         found = 0             /* Flag the string as not found in the record */
         read_size = size    /* Make sure we get all the way out of the loop */
         iterate s     /* Only need to find it once to know is needs replace */
        end
      say left(' ',2) string 'Found At Character Location',         /* Write */
          right(read_size+rposition,8)
      if position-10 < 1 then bstart = 1          /* Set beginning of buffer */
       else bstart = position-10             /* 10 before location of string */
      if out = '' then say left(' ',length(string)+3),  /* Check replacement */
             '-->'translate(substr(work,bstart,40),' ','15'x)'<--'  /* Write */
       else do                                /* A replacement was specified */
         save = work                                     /* Save what is was */
         work = substr(save,1,position-1),                 /* Replace string */
                ||out||substr(save,position+length(string))
         say left(' ',length(string)+3),                            /* Write */
             translate(substr(save,bstart,20+inlen),' ','15'x),
             ||'<-- is now -->'||,
             translate(substr(work,bstart,20+outlen),' ','15'x)
         found = 1                         /* We found and replaced a string */
        end
      offset = position + 1                /* Get past what we already found */
      roffset = rposition + 1              /* Get past what we already found */
     end
     if found then line = work                /* Replace buffer with updates */
/* In case the string spans buffers                                          */
     offset = length(line)-length(string)              /* Offset into buffer */
     if offset-1 > 0 then do                       /* Can search this buffer */
        line_out = substr(line,1,offset-1)                  /* Output buffer */
        start = offset             /* Start at last possible string location */
       end
      else do                                /* This is a 'too short' buffer */
        line_out = substr(line,1,length(line)-1)            /* Output buffer */
        start = length(line)               /* Start at the end of the buffer */
       end
    end
   otherwise do                       /* String may be split between buffers */
     offset = length(line)-length(string)              /* Offset into buffer */
     if offset-1 > 0 then do                       /* Can search this buffer */
        line_out = substr(line,1,offset-1)                  /* Output buffer */
        start = offset             /* Start at last possible string location */
       end
      else start = 1           /* This is 'too short' buffer-start beginning */
    end
  end
  call across_buffers                    /* In case the string spans buffers */
  if out ^= '' then do                              /* Check for replacement */
     'write' out_fd 'line_out' length(line_out)
     if rc ^= 0 | retval = -1 then do                    /* Check for errors */
        say 'Error Writing Output File.',                           /* Write */
            'RC='errno', Reason Code:' errnojr
        call exit rc'~File='file'.updated'                  /* Report errors */
       end
    end
  read_size = read_size + buffer_size                 /* Amount of data read */
 end
return

/*****************************************************************************/
/* In case the string spans buffers                                          */
/*****************************************************************************/
across_buffers:
 do readpos = start by 1 until readpos >= length(line)        /* Read buffer */
  character = substr(line,readpos,1)               /* Get the next character */
  characters = characters||character             /* Save the characters read */
  if character = substr(string,string_pos,1) then do          /* Check match */
     maybe = 1                        /* String may be split between buffers */
     if string_pos = length(string) then do            /* Check string match */
        bstart = read_size+(readpos-(length(string)-1))          /* Location */
        if generate ^= '' then do                      /* Generate specified */
           call add_repl_file       /* Add entry to replacement control file */
           found = 0           /* Flag the string as not found in the record */
           return      /* Only need to find it once to know is needs replace */
          end
        say left(' ',2) string 'Found At Character Location' right(bstart,8)
        if out = '' then say left(' ',length(string)+3) '-->'string'<--'
         else do
           line_out = line_out||out                         /* Output buffer */
           say left(' ',length(string)+3) string'<-- is now -->'out /* Write */
           found = 1                       /* We found and replaced a string */
          end
        string_pos = 1            /* re-Initialize string character position */
        characters = ''                 /* re-Initialize matching characters */
       end
      else string_pos = string_pos + 1                 /* No match-increment */
    end
   else do                                             /* No character match */
     if out ^= '' then,                             /* Check for replacement */
        line_out = line_out||characters                     /* Output buffer */
     maybe = 0                                                  /* Not split */
     string_pos = 1               /* re-Initialize string character position */
     characters = ''                    /* re-Initialize matching characters */
    end
 end
return

/*****************************************************************************/
/* Close the file                                                            */
/*****************************************************************************/
close_file:
 if in_fd ^= '' then do                             /* Input file was opened */
    'f_closfd' in_fd in_fd                               /* Close input file */
     if rc ^= 0 | retval = -1 then do                    /* Check for errors */
        say 'Error Closing Input File.',                            /* Write */
            'RC='errno', Reason Code:' errnojr
        say 'File='file                                     /* Report errors */
        set_rc = 4                              /* Set a warning return code */
       end
    in_fd = ''                       /* Initialize the input file descriptor */
   end
 if out_fd ^= '' then do                           /* Output file was opened */
    'f_closfd' out_fd out_fd                            /* Close output file */
     if rc ^= 0 | retval = -1 then do                    /* Check for errors */
        say 'Error Closing Output File.',                           /* Write */
            'RC='errno', Reason Code:' errnojr
        say 'File='file'.updated'                           /* Report errors */
        set_rc = 4                              /* Set a warning return code */
       end
    out_fd = ''                     /* Initialize the output file descriptor */
   end
 if ^found then do                                      /* Nothing was found */
    'unlink' file'.updated'                        /* Delete the output file */
    errno = 0                       /* We don't care if the delete failed... */
    errnojr = 0                 /* ...because it may not have existed anyway */
   end
  else do                                         /* OK, we found the string */
    in_process = 1         /* We are in process of creating the updated file */
    'unlink' file'.sasold'                 /* Make sure there is no old file */
    errno = 0                       /* We don't care if the delete failed... */
    errnojr = 0                 /* ...because it may not have existed anyway */
    'rename' file file'.sasold'               /* Rename the file to old file */
    if rc ^= 0 | retval = -1 then do                     /* Check for errors */
       say 'Error Renaming Input File to ".sasold".',               /* Write */
           'RC='errno', Reason Code:' errnojr
       call exit rc'~File='file                             /* Report errors */
      end
    'rename' file'.updated' file              /* Rename the update file file */
    if rc ^= 0 | retval = -1 then do                     /* Check for errors */
       say 'Error Renaming Output File to Input File.',             /* Write */
           'RC='errno', Reason Code:' errnojr
       call exit rc'~File='file'.updated'                   /* Report errors */
      end
    'unlink' file'.sasold'              /* Finsished, we no longer need this */
    in_process = 0                          /* re-Initialize in_process flag */
   end
return

/*****************************************************************************/
/* Add an entry in the replacement control file                              */
/*****************************************************************************/
add_repl_file:
 say left(' ',2) string 'Found in File:' file                       /* Write */
 next = grcf.0                                                /* Array count */
 string = strip(string)                                     /* remove blanks */
 tmp = generate file'!'string                       /* Copy to work variable */
 do until tmp = ''                    /* May need to split into 80-byte recs */
  parse var tmp chunk 79 tmp                     /* 79 is really only 78 max */
  chunk = strip(chunk)                                      /* Remove blanks */
  next = next + 1                                        /* Next array count */
  if tmp ^= '' then grcf.next = chunk '~'          /* Need to split the line */
   else grcf.next = chunk                     /* It will all fit on one line */
 end
 grcf.0 = next                                        /* Set the array count */
 read_size = size                                  /* Only need to know once */
 work = ''                            /* Found once searching for the string */
return                                                   /* Return to caller */

/*****************************************************************************/
/* Initialize variables, etc.                                                */
/*****************************************************************************/
initialize:
 numeric digits 11         /* Make sure we can understand 'FFFFFFFF' in dec. */
 set_rc = 0                            /* Initialize the warning return code */
 errno = 0                                     /* Initialize the return code */
 errnojr = 0                                   /* Initialize the reason code */
 in_process = 0                                /* Initialize in_process flag */
 tag = ''                                              /* Initialize the tag */
 drop files.                                 /* Initialize the list of files */
 in_fd =                             /* Initialize the input file descriptor */
 out_fd = ''                        /* Initialize the output file descriptor */
 grcf.0 = 0                     /* Initialize replacement control file count */
 codeset = 'EBCDIC ascii'                         /* Initialize codeing type */
 ignore = 'sasold updated' /* Ignore termorary files created by this process */
 ignore = ignore 'jpg png gif tif ico'                      /* Ignore images */
 ignore = ignore 'jar zip tar'                        /* Ignore packed files */
 ignore = ignore 'exe dll'                     /* Ignore executable binaries */
 ignore = ignore 'ttd ttf ttc'                               /* Ignore fonts */
 ignore = ignore 'dwt fla ix spk swf tdb'        /* Other ??? = NOT SURE === */
 exclude = 'jproxy'                                   /* Exclude executables */
 characters = ''             /* Initialize characters matching search string */
 string_pos = 1            /* Initialize character position in search string */
 maybe = 0              /* Initialize string split between buffers indicator */
 inlen = length(string)         /* Initialize the length of the input string */
 outlen = length(out)          /* Initialize the length of the output string */
 call check_input                /* Check for errors in the input parameters */
return                                                   /* Return to caller */

/*****************************************************************************/
/* Check the input parameters for errors                                     */
/*****************************************************************************/
check_input:
 if path = '' then call exit '16~Path|File Input Parameter Not Specified'
 if string = '' then call exit '16~Search String Input Parameter Not Specified'
 if out = string then,                    /* Changing to same thing: useless */
    call exit '0~Replacement String Input Parameter Same as Search String'
return                                                   /* Return to caller */

/*****************************************************************************/
/* Read the Customer Control file                                            */
/*****************************************************************************/
read_cust:
 parse arg function                                     /* Get function flag */
 drop gccf.                                        /* Make sure we are clean */
 'readfile rplvarin gccf.'                 /* Read the customer control file */
 if rc ^= 0 | errno ^= 0 | errnojr ^= 0 | retval = -1 then,  /* Check the RC */
    call exit rc'~Error Reading Control File: rplvarin'             /* Error */
 if function = 'init' then do                            /* Running generate */
    say ''                                                          /* Write */
    say 'Customer Control File:'                                    /* Write */
   end
 parms = ''                             /* Initialize the command parameters */
 do i = 1 to gccf.0                /* Read all customer control file records */
  if function = 'init' then say gccf.i                   /* Running generate */
  parse var gccf.i key 2 .                       /* See if this is a comment */
  gccf.i = strip(gccf.i)        /* Remove all preceeding and trailing blanks */
  if key ^= '*' then do                                       /* non-Comment */
     if substr(gccf.i,length(gccf.i),1) = '~' then,             /* Continued */
        parms = parms||substr(gccf.i,1,length(gccf.i)-2)          /* Portion */
      else do                                               /* Not continued */
        parms = parms||gccf.i                                  /* Parameters */
        parse var parms work_tag'=' work_gen .     /* Get tag and it's value */
        if work_gen = '' & function = '' then,   /* No tag value & ^generate */
           call exit '16~Missing Assignment for Tag:' work_tag'=,', /* Error */
                     'File: rplvarin'
        if wordpos(work_tag,'SASPATH ORIGINAL_SASPATH') ^= 0 then do /* Req. */
           parse var work_gen work_gen '<=' .  /* Strip end-of-line comments */
           work_gen = strip(work_gen)    /* Remove preceding/trailing blanks */
           if work_tag = 'SASPATH' then saspath = work_gen  /* SASPATH spec. */
            else orig_saspath = work_gen         /* ORIGINAL_SASPATH special */
          end
        c.work_tag = work_gen                         /* Make the assignment */
        parms = ''                      /* Initialize the command parameters */
       end
    end
 end
 if symbol('saspath') = 'LIT' & function = '' then, /* SASPATH was not found */
    call exit '16~Missing Tag: SASPATH=, File: rplvarin'
 if symbol('orig_saspath') = 'LIT' then,   /* ORIGINAL_SASPATH was not found */
    call exit '16~Missing Tag: ORIGINAL_SASPATH=, File: rplvarin'
 if orig_saspath = '' then,                  /* ORIGINAL_SASPATH was not set */
    call exit '16~Missing Assignment for Tag: ORIGINAL_SASPATH=, File: rplvarin'
 return                                                  /* Return to Caller */

/*****************************************************************************/
/* Initialize the output files                                               */
/*****************************************************************************/
init_files:
 say ''                                                             /* Write */
 say 'Initializing the Target USS Replacement Control File'         /* Write */
 drop grcf.                                        /* Make sure we are clean */
 grcf.1 = '**********************************************************'
 grcf.2 = '* Control File for Files Requiring Site Customization    *'
 grcf.3 = '* ----- Do NOT Alter this File ------------------------- *'
 grcf.4 = '**********************************************************'
 grcf.5 = '*'
 grcf.0 = 5                                                 /* Set the count */
return

/*****************************************************************************/
/* Read the SAS control file                                                 */
/*****************************************************************************/
read_sas:
 tag = ''                        /* Initialize to populate tag from RPLSASIN */
 drop gscf.                                        /* Make sure we are clean */
 rc = outtrap('NULL.')                      /* Set up to trap the TSO output */
 'readfile rplusasi gscf.'                      /* Read the SAS control file */
 if rc ^= 0 | errno ^= 0 | errnojr ^= 0 | retval = -1 | gscf.0 = 0 then,
    call exit rc'~Error Reading Control File: rplusasi'
 say ''                                                             /* Write */
 say 'SAS Control File:'                                            /* Write */
 parms = ''                             /* Initialize the command parameters */
 do i = 1 to gscf.0                          /* Process the SAS control file */
  gscf.i = strip(gscf.i)        /* Remove all preceeding and trailing blanks */
  say gscf.i                                                        /* Write */
  parse var gscf.i key 2 .                       /* See if this is a comment */
  if key ^= '*' then do                                       /* non-Comment */
     if substr(gscf.i,length(gscf.i),1) = '~' then,             /* Continued */
        parms = parms||substr(gscf.i,1,length(gscf.i)-2)          /* Portion */
      else do                                               /* Not continued */
        parms = parms||gscf.i                                  /* Parameters */
        parse var parms path'!' sstring'!' sout     /* path, search, replace */
        call check_input                    /* Check that the input is valid */
        if translate(substr(sout,1,12)) = 'SASGENERATE=' then do /* Generate */
           if length(sout) < 13 then,                         /* Missing tag */
              call exit '16~SASGENERATE Tag Missing, DDName='sas_ctl
           generate = translate(substr(sout,13))                  /* Get tag */
           if symbol('c.generate') = 'LIT' then, /* Tag missing - cust. file */
              call exit '16~Tag:' generate 'Missing from Customer Control File'
           tag = tag generate sstring';'       /* Save tag and search string */
          end
         else do                                /* This is an invalid record */
           say 'Invalid Record:' gscf.i', DDName='sas_ctl           /* Write */
           set_rc = 4                           /* Set a warning return code */
          end
        parms = ''                      /* Initialize the command parameters */
       end
    end
 end
 rc = outtrap('OFF')                                /* Turn the TSO trap off */
return

/*****************************************************************************/
/* Build the input for full replacement and call REPLUSS to replace strings  */
/*****************************************************************************/
do_replace:
 drop grcf.                                        /* Make sure we are clean */
 'readfile rplufile grcf.'                                  /* Read the file */
 if rc ^= 0 | errno ^= 0 | errnojr ^= 0 | retval = -1 then,  /* Check the RC */
    call exit rc'~Error Reading Replacement Control File: rplufile' /* Error */
 say ''                                                             /* Write */
 say 'Replacement Control File:'                                    /* Write */
 do i = 1 to grcf.0              /* Read the string replacement control file */
  say grcf.i                                                        /* Write */
 end
 parms = ''                             /* Initialize the command parameters */
 do i = 1 to grcf.0              /* Read the string replacement control file */
  parse var grcf.i key 2 .                       /* See if this is a comment */
  grcf.i = strip(grcf.i)        /* Remove all preceeding and trailing blanks */
  if key ^= '*' then do                                       /* non-Comment */
     grcf.i = strip(grcf.i)                                 /* Remove blanks */
     if substr(grcf.i,length(grcf.i),1) = '~' then,             /* Continued */
        parms = parms||substr(grcf.i,1,length(grcf.i)-2)          /* Portion */
      else do                                               /* Not continued */
        parms = parms||grcf.i                                  /* Parameters */
        parse var parms tag path'!'string      /* Get replacement parameters */
        address 'SH' './repluss' path'!'string'!'c.tag                /* Run */
        if rc ^= 0 then,                                            /* Error */
           call exit rc'~Error Executing ./repluss' path'!'string'!'c.tag
        parms = ''                      /* Initialize the command parameters */
       end
    end
 end
return                                                   /* Return to caller */

/*****************************************************************************/
/* Display any error message, diagnostics, and return to caller              */
/*****************************************************************************/
exit:
 parse arg retcode '~' message            /* Parse line - get RC and message */
 if message ^= '' then say message                   /* Display if specified */
 if errno ^= 0 & errnojr ^= 0 then do                    /* Check for errors */
    say 'Diagnosis:'                                                /* Write */
    'strerror' errno errnojr 'err.'           /* Get error description, etc. */
    do i = 1 to err.0           /* Loop until error descriptions all written */
     say err.i                                    /* Write error description */
    end
    if retcode = 0 then retcode = '1111'           /* Set a return code 1111 */
   end
 if in_process then do    /* We found an error in process of updating a file */
    say 'Restoring' file 'to its original state'                    /* Write */
    'unlink' file'.updated'                           /* Delete the new file */
    'rename' file'.sasold' file               /* Rename the old file to file */
   end
 if grcf.0 ^= 0 then do    /* A replacement control file needs to be created */
    'writefile rplufile 755 grcf.'                         /* Write the file */
    if rc ^= 0 | errno ^= 0 | errnojr ^= 0 | retval = -1 then do /* Check RC */
       say 'Error Writing the Replacement Control File rplufile'    /* Error */
       exit rc                                           /* Return to caller */
      end
   end
 say ''                                     /* Separate from previous output */
 if set_rc > retcode then retcode = set_rc           /* Warning has been set */
exit retcode                                             /* Return to caller */
