/* Rexx                                                                      */
/*****************************************************************************/
/*  PRODUCT: MVS SAS                                                         */
/*  PURPOSE: Tapedump process to replace values in a PDS                     */
/*  FROM: SAS INSTITUTE INC., SAS Campus Drive, Cary, NC 27513               */
/*****************************************************************************/
/* Changed 10Nov2010 RVC - Corrected SuperC truncated search string output   */
/* Changed 06Dec2010 RVC - Check RC after "REPLPDS" dsn','string','c.tag     */
/* Changed 06Dec2010 RVC - Corrected parsing of SuperC search string output  */
/* Changed 11Jan2011 GMK - Added Options Parms                               */
/* Changed 13Jan2011 GMK - Fixed bug so code can run against cntl file       */
/* Changed 31Jan2011 GMK - Add strip function to arfs in add_repl_file       */
/* Changed 14Mar2011 RVC - Pass options to REPLPDS call                      */
/* Changed 15Mar2011 RVC - Corrected parsing of ORIGINAL_xxx in read_cust:   */
/* Changed 16Mar2011 GMK - Fixed traceopt and concatenation of options parm  */
/* Changed 21Mar2011 RVC - Set Customer Replacement values for JOBCARD1-5    */
/* Changed 12Apr2011 GMK - Changed length(orig_hlq)+1) to ...+2) per RVC     */
/* Changed 19May2011 RVC - Permit quote (') in SuperC string searches        */
/* Changed 16Jun2011 RVC - Corrected return from skipping non-PO datasets    */
/*****************************************************************************/
/* Find a String in Members of PDSs and Optionally Replace it                */
/* Format:                                                                   */
/*  REPLPDS <dslevel | dsname | dsname(member)>,search_string                */
/*          ,<replace_string | SASGENERATE=<SAS_INIT | REPLACEMENTS | tag  > */
/*          ,<_OPT_ <IGNR ingorelist> <TRACE traceopt> > >                 > */
/*  Where:                                                                   */
/*   dslevel        = Dataset Level - ie TDI.MVS.SAS93                       */
/*   dsname         = Dataset Name - ie TDI.MVS.SAS93.CLIST                  */
/*   dsname(member) = Dataset Member - ie TDI.MVS.SAS93.CLIST(CLSTDBW0)      */
/*   search_string  = String to search for (case sensitive)                  */
/*   replace_string = Relacement for search_string (case sensitive)          */
/*   SASGENERATE=   = Keyword for Generating of Control Files                */
/*   SAS_INIT       = Keyword for Initialization of Control Files            */
/*   REPLACEMENTS   = Keyword for Changing strings given in Control Files    */
/*   tag            = Keyword for Tagging information in Control Files       */
/*   _OPT_          = processing options:IGNR ingorelist TRACE traceopts     */
/* Format for initializing file:                                             */
/*  REPLPDS dummy,dummy,SASGENERATE=SAS_INIT                                 */
/* Format for creating customer modifyable tags:                             */
/*  REPLPDS <dslevel | dsname>,search_string,SASGENERATE=tag                 */
/*  Example: REPLPDS TDI.MVA.SAS93,TDI.SAS93.,SASGENERATE=HLQ                */
/* Format for replacing strings:                                             */
/*  REPLPDS <dslevel | dsname | dsname(member)>,search_string,replace_string */
/*  Example: REPLPDS TDI.MVA.SAS93,TDI.SAS93.,ABC.SAS93.                     */
/* Format for replacing strings using the Control Files                      */
/*  REPLPDS dummy,dummy,SASGENERATE=REPLACEMENTS                             */
/* Notes:                                                                    */
/*  If DDNAME RPLSASIN is found at SASGENERATE=SAS_INIT then REPLPDS will    */
/*  use the file's tag and search_string to generate the control files       */
/*  instead of having to run REPLPDS SASGENERATE=tag separately for each tag */
/*  and search_string.                                                       */
/*****************************************************************************/

/* parse options  */
parse arg repprms '_OPT_' options
if options = " " then ,
  do
    traceopt = 'OFF'
    ignore = 'TAPECTL SASCNTL'
  end

else ,
  do
    parse UPPER var options before 'TRACE' traceparm after
    options = before || ' ' || after
    if strip(traceparm) = 'ON' then traceopt = 'ON'

    parse var options IGNR ignorlst
    if ignorlst = " " then nop
    else ignore = strip(ignorlst)
  end

  if traceopt = 'ON' then trace r

parse var repprms dsn'!' string'!' out  /* Get DSN, srch string, replacement */
out = strip(out)                                   /* Remove trailing blanks */
cust_ctl = 'RPLVARIN'                    /* DDNAME for customer control file */
sas_ctl = 'RPLSASIN'                          /* DDNAME for SAS control file */
repl_ctl = 'RPLFILES'  /* DDNAME for SAS target files for string replacement */
mem_dd = 'FINDIN'                    /* DDNAME for the member being searched */
call initialize                                /* Initialize variables, etc. */
if translate(substr(out,1,12)) = 'SASGENERATE=' then do   /* Generate wanted */
   if length(out) < 13 then,                                  /* Missing tag */
      call exit '16~SASGENERATE Tag Missing'
   generate = translate(substr(out,13))                           /* Get tag */
   tag = generate string';'                        /* Set the tag for search */
   out = ''                                               /* Force no output */
   select
    when generate = 'SAS_INIT' then do        /* Request to initialize Files */
      call init_files                         /* Initialize the output files */
      call read_sas                             /* Read the SAS control file */
      call build_file                             /* Build the RPLFILES file */
      call exit '0~Execution Successful'                       /* Go get out */
     end
    when generate = 'REPLACEMENTS' then do     /* Request to replace strings */
      call read_cust                       /* Read the Customer Control file */
      call do_replace                               /* Do a full replacement */
      call exit '0~Execution Successful'                       /* Go get out */
     end
    otherwise out = ''                       /* Blank out the replace string */
   end
  end
 else do                                     /* We may be replacing someting */
   if pos(string,out) ^= 0 then,      /* Search string in replacement string */
   call exit '16~Search String ('string') Within Replacement String('out').',
             'This Would Cause an Endless Loop.'                    /* Error */
  end
if tag = '' then tag = 'sas_single_search' string';'  /* Set customer search */
  else call read_cust                          /* Read Customer control File */
call listdsname                                       /* List the dataset(s) */
call read_dataset                                /* Read each dataset listed */
call exit '0~Execution Successful'         /* Return to caller - all is well */

/* List the dataset name(s)                                                  */
listdsname:
 parse var dsn dsn '(' single_member ')' . /* See if single member specified */
 skip = 0                                         /* Potential for searching */
 if single_member ^= '' then do                   /* Single member specified */
    member.0 = 3                                            /* Set variables */
    member.1 = dsn                                       /* Set the PDS name */
    member.2 = '--MEMBERS--'               /* Following is the single member */
    member.3 = single_member                            /* The single member */
    return                                               /* Return to caller */
   end
 stem = outtrap('MEMBER.')                  /* Set up to trap the TSO output */
 "LISTDS" "'"dsn"'" "MEMBERS"                          /* Try for single PDS */
 mrc = rc                                            /* Save the return code */
 "LISTDS" "'"dsn"'" "MEMBERS LEVEL"       /* List the DSNs and their members */
 if mrc ^= 0 & rc ^= 0 then do                              /* ListDS failed */
    if rc ^= 0 then do                                /* ListDS still failed */
       do i = 1 to member.0      /* For every record in output error display */
        say member.i                                                /* Write */
       end
       call exit '4~Error Executing LISTDS, DSN='dsn
      end
   end
 rc = outtrap('OFF')                                /* Turn the TSO trap off */
return                                                   /* Return to caller */

/* Read the list of datasets and their members                               */
read_dataset:
 found = 0                                   /* Initialize string found flag */
 in_process = 0                                /* Initialize in_process flag */
 do i = 1 to member.0                     /* For every line output by LISTDS */
  member.i = strip(member.i)                        /* Remove leading blanks */
  select
   when pos('.',member.i) ^= 0 then do             /* Found the dataset name */
     dataset_name = member.i                        /* Keep the dataset name */
     llq = substr(dataset_name,lastpos('.',dataset_name)+1) /* Get last qual */
     start = 0                            /* Re-Initialize member start flag */
    end
   when substr(member.i,1,8) = '--RECFM-' then do      /* DCB coming up next */
     skip = 0                                     /* Potential for searching */
     i = i + 1                               /* Increment to DCB information */
     parse var member.i recfm . . dsorg .             /* Get DCB information */
     if dsorg ^= 'PO' then skip = 1                       /* Not partitioned */
     parse var recfm key 2 .                     /* First character of recfm */
     if pos(key,'FV') = 0 then skip = 1            /* RECFM not F[B] or V[B] */
    end
   when skip then iterate i                             /* Get the next line */
   when substr(member.i,1,13) = 'THE FOLLOWING' then iterate /* Ignore these */
   when substr(member.i,1,6) = 'ALIAS(' then iterate         /* Ignore these */
   when start then do                         /* Valid member name(s) follow */
     parse var member.i member.i .               /* Exclude etc. information */
     work_member = strip(left(member.i,7))||'@'    /* Init. work member name */
     call search_dsn                           /* Search the dataset members */
     if out ^= '' & found then call replace_string      /* Replace specified */
    end
   when member.i = '--MEMBERS--' then do             /* Start of the members */
     say ' '                                                        /* Write */
     say 'DSN='dataset_name                                         /* Write */
     start = 1                                       /* Start of the members */
     if wordpos(llq,ignore) ^= 0 then do              /* Ignore this dataset */
        skip = 1                               /* Do not search this dataset */
        iterate i                                        /*Get the next line */
      end
     if generate ^= '' then do                         /* Generate specified */
        say left('Tag',15) left('Member',8) 'Line# Matched Record'  /* Write */
        say left('-',15,'-') left('-',8,'-') right('-',5,'-') left('-',80,'-')
       end
      else do                                            /* Normal execution */
        say left('Member',8) 'Line# Matched Record'                 /* Write */
        say left('-',8,'-') right('-',5,'-') left('-',80,'-')       /* Write */
       end
    end
   otherwise nop           /* Nothing else to consider including PS datasets */
  end
 end
return                                                   /* Return to caller */

/* Search the dataset members for the string and optionally replace string   */
search_dsn:
 drop input.                                       /* Make sure we are clean */
 call allocate                         /* Allocate the member to be searched */
 "EXECIO * DISKR" mem_dd "(STEM INPUT. FINIS"             /* Read the member */
 if rc ^= 0 then do                                 /* Check the return code */
    say 'Error Reading' dataset_name'('member.i')'                  /* Write */
    set_rc = 4                                  /* Set a warning return code */
    return                                               /* Return to caller */
   end
 work_tag = tag                                  /* Put tag in work variable */
 do j = 1 by 1 until work_tag = ''             /* Search for all the strings */
  parse var work_tag generate string ';' work_tag  /* Get the tag and string */
  if generate = 'sas_single_search' then generate = ''    /* Customer search */
  do k = 1 to input.0                                    /* For every record */
   if pos(string,input.k) ^= 0 then do                /* Search string found */
      found = 1                    /* Flag the string as found in the record */
      if generate ^= '' then,                         /* A tag was specified */
         say left(generate,15) left(member.i,8) right(k,5) input.k  /* Write */
       else  say left(member.i,8) right(k,5) input.k                /* Write */
      if out ^= '' then do             /* A replacement string was specified */
         lrecl = length(input.k)                    /* Get the record length */
         temp = input.k                                      /* Copy to temp */
         position = 0                   /* Initialize where we start looking */
         do until pos(string,temp,position+1) = 0    /* Find all occurrences */
          position = pos(string,temp)          /* Get the position of string */
          temp = substr(temp,1,position-1),          /* Update the temp copy */
                 ||out||substr(temp,position+length(string))
         end
         input.k = substr(temp,1,lrecl)     /* Replace record - may truncate */
         say left(' ',8) right('---->',5) input.k                   /* Write */
        end
     end
  end
 end
 call deallocate                 /* De-allocate the member that was searched */
return                                                   /* Return to caller */

/* Replace the search string with the replacement string if specified        */
replace_string:
 stem = outtrap('NULL.')                    /* Set up to trap the TSO output */
 "RENAME '"dataset_name"("member.i")'",           /* Save the current member */
         "'"dataset_name"("work_member")'"
 if rc ^= 0 then do                                 /* Check the return code */
    do i = 1 to null.0             /* All diagnostic output from the command */
     say null.i                                                     /* Write */
    end
    call exit rc'~Error Renaming 'dataset_name'('member.i') to',    /* Error */
                dataset_name'('work_member')'
   end
 rc = outtrap('OFF')                                /* Turn the TSO trap off */
 in_process = 1          /* We are in process of creating the updated member */
 call allocate                         /* Allocate the member to be searched */
 "EXECIO * DISKW" mem_dd "(STEM INPUT. FINIS"               /* Output record */
 if rc ^= 0 then,                                   /* Check the return code */
    call exit rc'~Error Writing' dataset_name'('member.i')'         /* Error */
 call deallocate                                      /* Deallocate the file */
 stem = outtrap('NULL.')                    /* Set up to trap the TSO output */
 "DELETE '"dataset_name"("work_member")'"              /* Delete the old one */
 if rc ^= 0 then do                                 /* Check the return code */
    do i = 1 to null.0             /* All diagnostic output from the command */
     say null.i                                                     /* Write */
    end
    say 'Error Deleting' dataset_name'('work_member'). Delete Manually'
    set_rc = 4                                  /* Set a warning return code */
   end
 rc = outtrap('OFF')                                /* Turn the TSO trap off */
 found = 0                                /* re-Initialize string found flag */
 in_process = 0                             /* re-Initialize in_process flag */
return                                                   /* Return to caller */

/* Allocate the dataset member being searched                                */
allocate:
 "ALLOCATE FILE("mem_dd") DATASET('"dataset_name"("member.i")') SHR REUSE"
 if rc ^= 0 then do                                 /* Check the return code */
    say 'Error Allocating' dataset_name'('member.i')'               /* Write */
    set_rc = 4                                  /* Set a warning return code */
   end
return                                                   /* Return to caller */

/* DeAllocate the dataset                                                    */
deallocate:
 "FREE FILE("mem_dd")"                                /* Deallocate the file */
 if rc ^= 0 then do                                 /* Check the return code */
    say 'Error Freeing' dataset_name'('member.i')'                  /* Write */
    set_rc = 4                                  /* Set a warning return code */
   end
return                                                   /* Return to caller */

/* Initialize variables, etc.                                                */
initialize:
 dsn = translate(dsn)                         /* Translate DSN to Upper Case */
 set_rc = 0                                /* Initialize warning return code */
 start = 0                                   /* Initialize member start flag */
 in_process = 0                                /* Initialize in_process flag */
 generate = ''                            /* Initialize the no-generate flag */
 tag = ''                         /* Initialize the multiple tag/string flag */
/* ignore = 'TAPECTL SASCNTL' Last Level Qualifier of Install PDSs - ignore  */
 call check_input                /* Check for errors in the input parameters */
return                                                   /* Return to caller */

/* Check for errors in the input parameters                                  */
check_input:
 if dsn = '' then call exit '16~DSN Input Parameter Not Specified'
 if string = '' then call exit '16~Search String Input Parameter Not Specified'
 if out = string then do                  /* Changing to same thing: useless */
    say '=>'string                                     /* Display the string */
    call exit '0~Replacement String Specified is the Same as Search String'
   end
 if substr(dsn,1,1) ^= "'" then dsn = userid()'.'dsn /* Append UserID if ^ ' */
  else dsn = strip(dsn,'B',"'")                               /* Strip off ' */
return                                                   /* Return to caller */

/* Read the Customer Control file                                            */
read_cust:
 drop gccf.                                        /* Make sure we are clean */
 "EXECIO * DISKR" cust_ctl "(STEM GCCF. FINIS"              /* Input records */
 if rc ^= 0 then,                                   /* Check the return code */
    call exit rc'~Error Reading Control File, DDName='cust_ctl      /* Error */
 parms = ''                             /* Initialize the command parameters */
 do i = 1 to gccf.0                /* Read all customer control file records */
  parse var gccf.i key 2 .                       /* See if this is a comment */
  gccf.i = strip(gccf.i)        /* Remove all preceeding and trailing blanks */
  if key ^= '*' then do                                       /* non-Comment */
     if substr(gccf.i,length(gccf.i),1) = '~' then,             /* Continued */
        parms = parms||substr(gccf.i,1,length(gccf.i)-2)          /* Portion */
      else do                                               /* Not continued */
        parms = parms||gccf.i                                  /* Parameters */
        parse var parms work_tag'=' work_gen       /* Get tag and it's value */
        work_gen = strip(work_gen,'T')             /* Remove trailing blanks */
        if work_gen = '' then,                /* No tag value & not generate */
           call exit '16~Missing Assignment for Tag:' work_tag'=,', /* Error */
                     'DDName='cust_ctl
        if wordpos(work_tag,'HLQ ORIGINAL_HLQ') ^= 0 then do     /* Required */
           parse var work_gen work_gen '<=' .  /* Strip end-of-line comments */
           work_gen = strip(work_gen)    /* Remove preceding/trailing blanks */
           if work_tag = 'HLQ' then hlq = work_gen            /* HLQ special */
            else orig_hlq = work_gen                 /* ORIGINAL_HLQ special */
          end
        c.work_tag = work_gen                         /* Make the assignment */
        parms = ''                      /* Initialize the command parameters */
       end
    end
 end
 if symbol('hlq') = 'LIT' then,                         /* HLQ was not found */
    call exit '16~Missing Tag: HLQ=, DDName='cust_ctl
 if symbol('orig_hlq') = 'LIT' then,           /* ORIGINAL_HLQ was not found */
    call exit '16~Missing Tag: ORIGINAL_HLQ=, File: rplvarin'
 if orig_hlq = '' then,                          /* ORIGINAL_HLQ was not set */
    call exit '16~Missing Assignment for Tag: ORIGINAL_HLQ=, File: rplvarin'
 return                                                  /* Return to Caller */

/* Initialize the output files                                               */
init_files:
 say ''                                                             /* Write */
 say 'Initializing the Target PDS Replacement Control File'         /* Write */
 drop grcf.                                        /* Make sure we are clean */
 grcf.1 = '**********************************************************'
 grcf.2 = '* Control File for Files Requiring Site Customization    *'
 grcf.3 = '* ----- Do NOT Alter this File ------------------------- *'
 grcf.4 = '**********************************************************'
 grcf.5 = '*'
 grcf.0 = 5                                                 /* Set the count */
 say ''                                                             /* Write */
 say 'Initializing the Customer Control File'                       /* Write */
 drop gccf.                                        /* Make sure we are clean */
 gccf.1 = '**********************************************************'
 gccf.2 = '* Control File for Customizing SAS at Your Site          *'
 gccf.3 = '* Enter the Values Associated with the Tags Given Below  *'
 gccf.4 = "* Note: Continue long lines with ' ~' in columns 79-80   *"
 gccf.5 = '**********************************************************'
 gccf.6 = '*'
 gccf_hlq1 = 7                        /* Reserve line for the ORIGINAL_HLQ = */
 gccf_hlq2 = 8                  /* Reserve line for the ORIGINAL_HLQ comment */
 gccf_saspath1 = 9                /* Reserve line for the ORIGINAL_SASPATH = */
 gccf_saspath2 = 10         /* Reserve line for the ORIGINAL_SASPATH comment */
 gccf.0 = gccf_saspath2                                     /* Set the count */
return                                                   /* Return to caller */

/* Read the SAS control file                                                 */
read_sas:
 tag = ''                        /* Initialize to populate tag from RPLSASIN */
 drop gscf.                                        /* Make sure we are clean */
 stem = outtrap('NULL.')                    /* Set up to trap the TSO output */
 "EXECIO * DISKR" sas_ctl "(STEM GSCF. FINIS"               /* Input records */
 if rc ^= 0 then call exit '16~Error Reading Control File, DDName='sas_ctl
 say ''                                                             /* Write */
 say 'SAS Control File:'                                            /* Write */
 parms = ''                             /* Initialize the command parameters */
 do i = 1 to gscf.0                          /* Process the SAS control file */
  gscf.i = strip(gscf.i)        /* Remove all preceeding and trailing blanks */
  say gscf.i                                                        /* Write */
  parse var gscf.i key 2 .                       /* See if this is a comment */
  if key ^= '*' then do                                       /* non-Comment */
     if substr(gscf.i,length(gscf.i),1) = '~' then,             /* Continued */
        parms = parms||substr(gscf.i,1,length(gscf.i)-2)          /* Portion */
      else do                                               /* Not continued */
        parms = parms||gscf.i                                  /* Parameters */
        parse var parms dsn'!' string'!' out         /* DSN, search, replace */
        call check_input                    /* Check that the input is valid */
        if translate(substr(out,1,12)) = 'SASGENERATE=' then do  /* Generate */
           if length(out) < 13 then,                          /* Missing tag */
              call exit '16~SASGENERATE Tag Missing, DDName='sas_ctl
           generate = translate(substr(out,13))                   /* Get tag */
           s.string = generate            /* Reverse assignment for SAS_INIT */
           tag = tag generate string';'        /* Save tag and search string */
          end
         else do                                /* This is an invalid record */
           say 'Invalid Record:' gscf.i', DDName='sas_ctl           /* Write */
           set_rc = 4                           /* Set a warning return code */
          end
        parms = ''                      /* Initialize the command parameters */
       end
    end
 end
 rc = outtrap('OFF')                                /* Turn the TSO trap off */
return

/* Build the RPLFILES file                                                   */
build_file:
 say ''                                                             /* Write */
 say 'Searching for Strings'                                        /* Write */
 address 'ISPEXEC' 'LMDINIT LISTID(ID) LEVEL('dsn')'
 if rc ^= 0 then,                                   /* Check the return code */
    call exit rc'~Error Executing LMDINIT LISTID(ID) LEVEL('dsn')'  /* Error */
 "ALLOCATE FILE(SYSIN) UNIT(SYSALLDA) NEW TRACKS SPACE(1,1)",       /* SYSIN */
        "DELETE REUSE LRECL(80) RECFM(F B) BLKSIZE(3120)"
 if rc ^= 0 then,                                   /* Check the return code */
    call exit rc'~Error Allocating Dynamic SYSIN Dataset'           /* Error */
 queue 'SLIST ON'                                         /* List statements */
 "EXECIO 1 DISKW SYSIN"                                        /* Write LNCT */
 if rc ^= 0 then,                                   /* Check the return code */
    call exit rc'~Error Writing SLIST ON to SYSIN Dataset'          /* Error */
 work_tag = tag                                  /* Put tag in work variable */
 do until work_tag = ''                 /* All strings specified in RPLSASIN */
  parse var work_tag generate string ';' work_tag  /* Get the tag and string */
  temp = string                                       /* Move to a work area */
  sub = -1                                         /* Initialize the pointer */
  do until sub = 0                               /* Search the entire string */
   sub = pos("'",temp,sub+2)                        /* Increment the pointer */
   if sub > 0 then temp = substr(temp,1,sub)||"'"||substr(temp,sub+1) /* Add */
  end
  queue "SRCHFOR '"temp"'"                                  /* Search string */
  "EXECIO 1 DISKW SYSIN"                             /* Write SRCHFOR string */
  if rc ^= 0 then,                                  /* Check the return code */
     call exit rc'~Error Writing SRCHFOR' string 'to SYSIN Dataset' /* Error */
  call add_cust_file generate string                               /* Add it */
 end
 queue 'LNCT 999999'                                           /* Line count */
 "EXECIO 1 DISKW SYSIN (FINIS"             /* Write SLIST and close the file */
 if rc ^= 0 then,                                   /* Check the return code */
    call exit rc'~Error Writing SLIST to SYSIN Dataset'             /* Error */
 do until rrc ^= 0                               /* All files in the DSLevel */
  call get_dataset                                   /* Get the next dataset */
  if rrc = 0 & ((ZDLDSORG = 'PO' | ZDLDSORG = 'PO-E') & ZDLRECFM ^= 'U ') then,
     call search_for                    /* Good RC & PDS - search for string */
 end
 address 'ISPEXEC' 'LMDFREE LISTID('id')'                   /* Free the List */
 if rc ^= 0 then,                                   /* Check the return code */
    say 'Error Executing LMDFREE LISTID('id')'                      /* Write */
 say ''                                                             /* Write */
 "EXECIO * DISKR OUTDD (STEM RESULT. FINIS"         /* Read the results file */
 if rc ^= 0 then,                                   /* Check the return code */
    call exit rc'~Error Reading the Results Dataset'                /* Error */
 found = 0                                      /* Initialize the found flag */
 say 'Replacement Control File'                                     /* Write */
 do i = 1 to result.0                      /* All records in the result file */
 say '->'||strip(result.i,'T')                         /* Display the record */
  select
   when pos('"',result.i) ^= 0 then do          /* This is a "string" record */
     parse var result.i . '"' string 29 mem 37 . 51 finds recs .   /* Values */
     parse var string string '"' .                        /* Strip off quote */
     temp = tag                      /* Put tag/string in temporary location */
     if length(string) = 22 then do until temp = ''  /* All searched strings */
        parse var temp . arfg ';' temp   /* Get the next searched for string */
        if length(string) < length(arfg) then do           /* Possible match */
           if substr(arfg,1,length(string)) = string then string = arfg
          end
       end
     if datatype(finds) = 'NUM' & datatype(recs) = 'NUM' then do    /* Valid */
        found = 1                                      /* Set the found flag */
        call add_repl_file s.string dsn mem string                 /* Add it */
       end
    end
   when result.i = '' then found = 0       /* This is between member records */
   when pos('SRCH DSN:',result.i) ^= 0 then,         /* This is a DSN record */
     parse var result.i . 'SRCH DSN:' dsn .                    /* Get values */
   otherwise do                                    /* May be a member record */
     if found then do                             /* This is a member record */
        parse var result.i mem finds recs .                    /* Get values */
        if datatype(finds) = 'NUM' & datatype(recs) = 'NUM' then,   /* Valid */
           call add_repl_file s.string dsn mem string              /* Add it */
       end
    end
  end
 end
return                                                   /* Return to caller */

/* Get the dataset from the list                                             */
get_dataset:
 address 'ISPEXEC' 'LMDLIST LISTID('id') OPTION(LIST)',  /* Get the next dsn */
                   'DATASET(dsn) STATS(YES)'
 rrc = rc                                  /* Save RC for checking by caller */
 llq = substr(dsn,lastpos('.',dsn)+1)                  /* Get last qualifier */
 select
  when wordpos(llq,ignore) ^= 0 then ZDLDSORG = ''    /* Ignore this dataset */
  when rrc = 0 then,                                           /* Good to go */
    line = 'Dataset:' substr(dsn,1,44)', VolSer:' ZDLVOL,           /* Write */
         'Org:' ZDLDSORG', Type:' ZDLDSNTP', RecFM:' ZDLRECFM
  when rrc = 4 then,                                    /* Dataset not found */
    call exit '16~'dsn 'Not Found'                                  /* Error */
  when rrc = 8 then nop                                 /* We are at the end */
  otherwise,                                              /* Bad return code */
    call exit rrc'~Error Executing LMDLIST LISTID('id')'            /* Error */
 end
return                                                   /* Return to caller */

/* Search a single PS or an whole library                                    */
search_for:
 "ALLOCATE FILE(NEWDD) DATASET('"dsn"') SHR REUSE"   /* Allocate the dataset */
 if rc ^= 0 then,                                   /* Check the return code */
    call exit rc'~Error Allocating Dataset' dsn                     /* Error */
 address 'ISPEXEC' "SELECT PGM(ISRSUPC) PARM(SRCHCMP,NOSUMS,LMTO,XREF)"
 select
  when rc = 0 then say line 'No string found'
  when rc = 1 then say line 'String(s) found'
  otherwise say        line '? Error, RC=' rc
 end
return

/* Add target replacement file to the RPLFILES member                        */
add_repl_file:
 parse arg arfg arfd arfm arfs                   /* Get the input parameters */
 arfs = strip(arfs)                                         /* remove blanks */
 say arfg arfd'('arfm'),'arfs                                       /* Write */
 nextr = grcf.0                                               /* Array count */
 tmp = arfg "'"arfd"("arfm")'!"arfs                      /* Copy to work var */
 do until tmp = ''                    /* May need to split into 80-byte recs */
  parse var tmp chunk 79 tmp                     /* 79 is really only 78 max */
  chunk = strip(chunk)                                      /* Remove blanks */
  nextr = nextr + 1                                      /* Next array count */
  if tmp ^= '' then grcf.nextr = chunk '~'         /* Need to split the line */
   else grcf.nextr = chunk                    /* It will all fit on one line */
 end
 grcf.0 = nextr                                       /* Set the array count */
return                                                   /* Return to caller */

/* Add tag to the customer control file                                      */
add_cust_file:
 parse arg actag acorig                          /* Get the input parameters */
 if symbol('f.actag') = 'VAR' then return /* Not the first time for this tag */
 nextc = gccf.0                                               /* Array count */
 if actag = 'HLQ' then do                /* We need to save the original HLQ */
    gccf.gccf_hlq1 = 'ORIGINAL_HLQ='acorig '<== Do Not Change!!!'  /* Assign */
    gccf.gccf_hlq2 = '*'                                           /* Assign */
   end
 if actag = 'SASPATH' then do        /* We need to save the original SASPATH */
    gccf.gccf_saspath1 = 'ORIGINAL_SASPATH='acorig '<== Do Not Change!!!'
    gccf.gccf_saspath2 = '*'                                       /* Assign */
   end
 nextc = nextc + 1                                       /* Next array count */
 if length(acorig) > 35 then acorig = substr(acorig,1,32)'...'
 gccf.nextc = '* Add' actag 'replacement below. Replacing:' acorig
 nextc = nextc + 1                                       /* Next array count */
 select                                         /* Looking for JOBCARDx tags */
  when actag = 'JOBCARD1' then actagout =,                       /* JOBCARD1 */
       "//USERNMA  JOB ACCOUNT-CODE,'PROGRAMMER-NAME',USER=USERNM,"
  when actag = 'JOBCARD2' then actagout =,                       /* JOBCARD2 */
       "//  MSGCLASS=H,MSGLEVEL=(1,1),CLASS=S,"
  when actag = 'JOBCARD3' then actagout =,                       /* JOBCARD3 */
       "//  TIME=(20,00),REGION=96M"
  when actag = 'JOBCARD4' then actagout =,                       /* JOBCARD4 */
       "//* JOBCARD VALUES ARE EXAMPLES ONLY. THEY MAY NOT"
  when actag = 'JOBCARD5' then actagout =,                       /* JOBCARD5 */
       "//* WORK FOR YOU. SUPPLY SETTINGS FOR YOUR SITE."
  otherwise actagout = ''                              /* Not a JOBCARDx tag */
 end
 gccf.nextc = actag'='actagout                /* It will all fit on one line */
 gccf.0 = nextc                                       /* Set the array count */
return                                                   /* Return to caller */

/* Build the input for full replacement and call REPLPDS to replace strings  */
do_replace:
 drop grcf.                                        /* Make sure we are clean */
 "EXECIO * DISKR" repl_ctl "(STEM GRCF. FINIS"              /* Input records */
 if rc ^= 0 then call exit rc'~Error Reading Control File, DDName='repl_ctl
 say ''                                                             /* Write */
 say 'Replacement Control File:'                                    /* Write */
 do i = 1 to grcf.0              /* Read the string replacement control file */
  say grcf.i                                                        /* Write */
 end
 parms = ''                             /* Initialize the command parameters */
 do i = 1 to grcf.0              /* Read the string replacement control file */
  parse var grcf.i key 2 .                       /* See if this is a comment */
  grcf.i = strip(grcf.i)        /* Remove all preceeding and trailing blanks */
  if key ^= '*' then do                                       /* non-Comment */
     grcf.i = strip(grcf.i)                                 /* Remove blanks */
     if substr(grcf.i,length(grcf.i),1) = '~' then,             /* Continued */
        parms = parms||substr(grcf.i,1,length(grcf.i)-2)          /* Portion */
      else do                                               /* Not continued */
        parms = parms||grcf.i                                  /* Parameters */
        parse var parms tag dsn'!'string   /* Get the replacement parameters */
        dsn = "'"hlq||substr(dsn,length(orig_hlq)+2)  /* Set to customer HLQ */
        "REPLPDS" dsn'!'string'!'c.tag '_OPT_' options                /* Run */
        if rc ^= 0 then,                                            /* Error */
           call exit rc'~Error Executing "REPLPDS"' dsn'!'string'!'c.tag,
                     '_OPT_' options
        parms = ''                      /* Initialize the command parameters */
       end
    end
 end
return                                                   /* Return to caller */

/* Exit                                                                      */
exit:
 parse arg retcode '~' message            /* Parse line - get RC and message */
 if message ^= '' then say message '(RC='retcode')'  /* Display if specified */
 stem = outtrap('NULL.')                    /* Set up to trap the TSO output */
 if in_process then do    /* We found an error in process of updating member */
    say 'Restoring' member.i 'to its original state'
    "DELETE '"dataset_name"("member.i")'"           /* Delete the new member */
    "RENAME '"dataset_name"("work_member")'", /* Restore the original member */
            "'"dataset_name"("member.i")'"
   end
 "FREE FILE("mem_dd")"                                /* Deallocate the file */
 rc = outtrap('OFF')                                /* Turn the TSO trap off */
 if grcf.0 ^= 0 then do    /* A replacement control file needs to be created */
    "EXECIO * DISKW" repl_ctl "(STEM GRCF. FINIS"           /* Output record */
    if rc ^= 0 then do                              /* Check the return code */
       say 'Error Writing Control File, DDName='repl_ctl            /* Error */
       exit rc                                           /* Return to caller */
      end
   end
 if gccf.0 ^= 0 then do       /* A customer control file needs to be created */
    "EXECIO * DISKW" cust_ctl "(STEM GCCF. FINIS"           /* Output record */
    if rc ^= 0 then do                              /* Check the return code */
       say 'Error Writing Control File, DDName='cust_ctl            /* Error */
       exit rc                                           /* Return to caller */
      end
   end
 say ''                                     /* Separate from previous output */
 if set_rc > retcode then retcode = set_rc           /* Warning has been set */
exit retcode                                             /* Return to caller */
