/*
 * LASRTableMetadataToDataSet
 * 
 * Purpose: Takes a LASR table, finds it in metadata, and loads associated metadata to a data set.
 *
 * Parms:
 * 		data= The output data set to create.  If unspecified, defaults to work.lasrtbl.
 *		name= The name of the LASR table to retrieve.  This name must be unique in the metadata server.
 *
 */
/* ############################################################################## */
/* Copyright (c) 2013-2025, SAS Institute Inc., Cary, NC, USA. All Rights Reserved. */
/* ############################################################################## */
%macro LASRLibraryMetadataToDataSet(data=WORK.LASRLIB, name=);

	%if ("&name." eq "" ) %then %do;
	   %msg( type=ERROR,msg=%str(Library name must be specified with NAME=));
	   %return;
	%end;

	%let query = %str(omsobj:SASLibrary?@Engine contains 'SASIOLA' and @Name=%quote(%')&name.%quote(%'));

	data &data.;

	/* The KEEP statement defines the variables to include in the output data set.  */
	  keep libid libname libengine libref libtag smp servname lasrhost lasrport lasrinst path signer 
		   tablemem externalmem lifetime force nodes
		   loggingenabled loggingpath loggingmaxfilesize loggingmaxrollover loggingkeeplog loggingadditional
           serververbose;

	/* The LENGTH statement defines variables for function arguments and
	assigns the maximum length of each variable.  */

	  length 
	   		libid $17

			/* Library properties */
			libname $128 
			libengine
			libref $8
			libtag $32

			/* Server properties */
			smp $8
			lasrhost $256 
			lasrport $8 
			signer 
			lasrinst 
			path $256
			force $5
			nodes $8

			tablemem 
			externalmem $3
			lifetime			$32

			loggingenabled $8
			loggingpath $256
			loggingmaxfilesize
			loggingmaxrollover  
			loggingkeeplog		$8
			loggingadditional	$128

			serververbose	$8

			liburi 
			connuri

			upasnuri 
			propnuri
			servnuri 
			servname $256
			type $32

			tmp $256
			;

	/* The CALL MISSING routine initializes the output variables to missing values.  */
	  call missing(lasrinst,path,liburi,servnuri,servname,propnuri,connuri,upasnuri,libname,libengine,libref,lasrhost,lasrport,libtag,tmp,signer,smp,force,nodes);
	  call missing(tablemem,externalmem,lifetime,loggingenabled,loggingpath,loggingmaxfilesize,loggingmaxrollover,loggingkeeplog,loggingadditional,serververbose);

	  /* Query for the Library Object */
	  nlibobj=1;
	  librc=metadata_getnobj("&query.",nlibobj,liburi);
	  if (librc <1) then put librc " matching libraries found.";

	  /* Process the library object */
	  do while (librc>0);

	     /* Get Library attributes */
	     rc=metadata_getattr(liburi,'Name',libname);
	     rc=metadata_getattr(liburi,'Engine',libengine);
		 rc=metadata_getattr(liburi,'Libref',libref);
		 rc=metadata_getattr(liburi,'Id',libid);
		 put libengine "library " libname;
		 put @7 "LIBREF=" libref;	 

		 /* Get Library properties */
		 n=1;
		 proprc=metadata_getnasn(liburi,'Properties',n, propnuri);
		 do while (proprc>0);
	    if proprc > 0 then do;
	       call missing(type,libid);
	       rc=metadata_resolve(propnuri,type,libid);

	       if type='Property' then do;
		   	  rc=metadata_getattr(propnuri,'Name',tmp);
			  if (tmp='Library.LASR.Property.Tag.Name.xmlKey.txt') then do;
		      	rc=metadata_getattr(propnuri,'DefaultValue',libtag);
			    put @7 "TAG=" libtag;
			  end;
			  else
			  	put "NOTE: Unprocessed Library property " tmp;
		   end; 

	       n+1;
	       proprc=metadata_getnasn(liburi,'Properties',n,propnuri);
		 end; /* if proprc > 0 */
		end; /* do while (proprc>0) */

		/* Get Library's LASR Server Connection */
	    uprc=metadata_getnasn(liburi,'LibraryConnection',1,upasnuri);
		if uprc > 0 then do;

		/* When a LibraryConnection association is found, the METADATA_RESOLVE function 
		is called to resolve the URI to an object on the metadata server. */

	    call missing(type,libid,lasrhost,lasrport);
	    rc=metadata_resolve(upasnuri,type,libid);
		    
		/* Get Connection Attributes */
		if type='SASClientConnection' then do;

	      put type;
	      rc=metadata_getattr(upasnuri,'Port',lasrport);
	      put @7 "Port " lasrport;

			/* Get Connection Properties */
			n=1;
			proprc=metadata_getnasn(upasnuri,'Properties',n, propnuri);
			do while (proprc>0);
				if proprc > 0 then do;
					call missing(type,libid);
					rc=metadata_resolve(propnuri,type,libid);

					if type='Property' then do;
						rc=metadata_getattr(propnuri,'PropertyName',tmp);
						if (tmp='SIGNER') then do;
							rc=metadata_getattr(propnuri,'DefaultValue',signer);
							put @7 "SIGNER=" signer;
						end;
						else if (tmp='HOST') then do;
							rc=metadata_getattr(propnuri,'DefaultValue',lasrhost);
							put @7 "HOST=" lasrhost;
						end;
						else
							put "Unprocessed Connection property " tmp;
					end; 

					n+1;
					proprc=metadata_getnasn(upasnuri,'Properties',n,propnuri);
					end; /* if proprc > 0 */
				end; /* Property processing */

				/* Get Library's LASR Server */
				n=1;
			    servrc=metadata_getnasn(upasnuri,'Source',n,servnuri);
				if servrc > 0 then do;
				    call missing(type,smp);
				    rc=metadata_resolve(servnuri,type,libid);
					    
					/* Get Server Attributes */
					if type='ServerComponent' then do;

				      rc=metadata_getattr(servnuri,'Name',servname);
				      put type servname;

					  /* Get Server Properties */

						n=1;
						proprc=metadata_getnasn(servnuri,'Properties',n, propnuri);
						do while (proprc>0);
							if proprc > 0 then do;
								call missing(type,libid);
								rc=metadata_resolve(propnuri,type,libid);

								if type='Property' then do;
									rc=metadata_getattr(propnuri,'PropertyName',tmp);
									if (tmp='INSTALL') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',lasrinst);
										put @7 "INSTALL=" lasrinst;
									end;
									else if (tmp='PATH') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',path);
										put @7 "PATH=" path;
									end;
									else if (tmp='FORCE') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',force);
										put @7 "FORCE=" force;
									end;
									else if (tmp='NODES') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',nodes);
										put @7 "NODES=" nodes;
									end;
									else if (tmp='SMPMODE') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',smp);
										put @7 "SMP=" smp;
									end;
									else if (tmp='TABLEMEM') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',tablemem);
										put @7 tmp "=" tablemem;
									end;
									else if (tmp='EXTERNALMEM') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',externalmem);
										put @7 tmp "=" externalmem;
									end;
									else if (tmp='LIFETIME') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',lifetime);
										put @7 tmp "=" lifetime;
									end;
									else if (tmp='VERBOSE') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',serververbose);
										put @7 tmp "=" serververbose;
									end;
									else if (tmp='LOGGING') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',loggingenabled);
										put @7 tmp "=" loggingenabled;
									end;
									else if (tmp='LOGPATH') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',loggingpath);
										put @7 tmp "=" loggingpath;
									end;
									else if (tmp='MAXFILESIZE') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',loggingmaxfilesize);
										put @7 tmp "=" loggingmaxfilesize;
									end;
									else if (tmp='MAXROLLNUM') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',loggingmaxrollover);
										put @7 tmp "=" loggingmaxrollover;
									end;
									else if (tmp='KEEPLOG') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',loggingkeeplog);
										put @7 tmp "=" loggingkeeplog;
									end;
									else if (tmp='ADDTIONALLOGOPTIONS') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',loggingadditional);
										put @7 tmp "=" loggingadditional;
									end;
									else
										put "Unprocessed Server property " tmp;
								end; 

								n+1;
								proprc=metadata_getnasn(servnuri,'Properties',n,propnuri);
								end; /* if proprc > 0 */
							end; /* Property processing */
					end;
				end; /* servrc>0 */



			end; /* Connection processing */ 



		end; /* if uprc > 0 */

		  output;
		 /* Look for another library */
		 nlibobj+1;
	     librc=metadata_getnobj("&query.",nlibobj,liburi);
	  end; /* do while (librc>0) */
	run;

%mend;
