/*
 * ImportDLM
 *
 * Purpose: Imports a delimited file
 *
 * Parameters:
 *
 *		dlm=		The source dlm file (full path) to load to the target library
 *		table=		The target table name (pre-processed for eliminating special/prohibited characters)
 * 		libref=		The already-assigned libref into which to load the table
 *		getnames=	If YES, column names will be retrieved from first row.  Defaults to YES
 *		startrow=	The starting row from which data will be read.  Defaults to 2.
 *		guessrows=	The number of data rows to read in order to determine type and length for columns
 *		delim=		The delimiter to expect in the input file
 */
/* ############################################################################## */
/* Copyright (c) 2013-2025, SAS Institute Inc., Cary, NC, USA. All Rights Reserved. */
/* ############################################################################## */
%macro ImportDLM		( dlm=
                        , table=
						, libref=
						, getnames=YES
						, startrow=2
						, guessrows=500
						, delim=
						);

	%let dlm=%superq(dlm);
	%let table=%superq(table);
	%let delim=%superq(delim);

	/* Set the DBMS setting based upon file extension */
	data _null_;
		extension=upcase(scan("&dlm.",-1,'.'));
		if 		( extension eq "TSV" or extension eq "TAB" ) then
			call symput ('DBMS','TAB');
		else if ( extension eq "TXT" ) then
			call symput ('DBMS','DLM');
		else
			call symput ('DBMS',strip(extension));
	run;

	/* Delimiter implied in these DBMS value cases */
	%if 		( &DBMS. eq TAB or &DBMS. eq CSV ) %then %LET DELIM=;
	/* If delimited file type, we need to set the delimiter based on what was requested */
	%else %if 	( &DBMS. eq DLM ) %then %do; 
		/* Set the delimiter */
		%if 		(%str(&delim.) eq %str()
				or   %str(&delim.) eq %str(MISSING) ) 		%then 	%LET DELIM=%str(%'09%'x);
		%else %if 	(&delim. eq TAB or &delim. eq tab)	 	%then 	%LET DELIM=%str(%'09%'x);
		%else %if 	(&delim. eq SPACE or &delim. eq space)	%then 	%LET DELIM=%str(%' %');
		%else %if 	(%kindex(&delim.,%str(%')) ne 1) 		%then	%LET DELIM=%str(%'&delim.%');
	%end;

	%PUT NOTE: DBMS=&DBMS.;
	%PUT NOTE: DLM=&DLM.;
	%LET EFI_QUOTED_NUMERICS = yes;
	%LET EFI_MISSING_NUMERICS = yes;

	FILENAME impfile "%unquote(&dlm.)";

	/* Convert DLM file into SAS dataset */
	%setoption(on,fullstimer);
	%setoption(on,mprint);
    PROC IMPORT
      DATAFILE=impfile
      OUT=&libref.."%unquote(&table.)"n
      REPLACE
      DBMS=&DBMS;
	  %if ( %str(&DELIM.) ne %str()) %then %do;
	  	DELIMITER=%unquote(&DELIM.);
	  %end;
      GETNAMES=&GETNAMES.;
	  DATAROW=&STARTROW.;
	  GUESSINGROWS=&GUESSROWS.; /* Could be ALL to use all rows */
    RUN;
	%setoption(restore,mprint);
	%setoption(restore,fullstimer);

%mend;
