/*
 * FilterReloadOnStartLibraries
 * 
 * Purpose: Takes a data set produced by GetLASRServerLibrariesList, along with reload-on-start settings.
 *			Produces a filtered list of the libraries to reload-on-start based upon the reload-on-start settings.
 *
 * Parms:
 * 		data= The input list of libraries from GetLASRServerLibrariesList.
 *		out = The output filtered list of tables.  If the same as input data, input data will be overwritten
 *			  with result.
 *		ros_enabled_var = The variable to check and set for reload-on-start status.
 *		dp_assigned_var = The variable to check which indicates if a data provider library is assigned.
 *
 * Notes:
 */
/* ############################################################################## */
/* Copyright (c) 2013-2025, SAS Institute Inc., Cary, NC, USA. All Rights Reserved. */
/* ############################################################################## */
%macro FilterReloadOnStartLibraries( data=, out=, ros_enabled_var=ros_enabled, dp_assigned_var=dp_assigned);

	%if ( "&DATA." eq "" ) %then %do;
	   %msg( type=ERROR,msg=%str(Input data must be specified with DATA=));
	   %return;
	%end;
	%if ( "&ROS_ENABLED_VAR." eq "" ) %then %do;
	   %LET ROS_ENABLED_VAR=ros_enabled;
	%end;

	%if ( "&OUT." eq "" ) %then %do;
		%let OUT=&DATA.;
	%end;

	data &OUT.;
	   set &DATA.;

	   if ( kupcase(&ROS_ENABLED_VAR.) eq 'YES' and kupcase(&DP_ASSIGNED_VAR.) eq 'YES' );
	run;

%mend;

/*
 * FilterReloadOnStartTables
 * 
 * Purpose: Takes a data set produced by GetMetadataTablesList, along with reload-on-start settings.
 *			Produces a filtered list of the tables to reload-on-start based upon the reload-on-start settings.
 *
 * Parms:
 * 		data= The input list of tables from GetMetadataTablesList.
 *		out = The output filtered list of tables.  If the same as input data, input data will be overwritten
 *			  with result.
 *		ros_lib_method = Input library's reload-on-start method setting.
 *		ros_enabled_var = The variable to check and set for reload-on-start status.
 *		ros_lib_tabledefault = Input library's reload-on-start table default setting.
 *
 * Notes:
 *		It is assumed that the input library has reload-on-start enabled.  
 *		It is assumed that the data provider library specified in fullref is assigned.
 */
%macro FilterReloadOnStartTables( data=, out=, ros_lib_method=, ros_lib_tabledefault=);

	%if ( "&DATA." eq "" ) %then %do;
	   %msg( type=ERROR,msg=%str(Input data must be specified with DATA=));
	   %return;
	%end;
	%if ( "&ROS_LIB_METHOD." eq "" ) %then %do;
	   %msg( type=ERROR,msg=%str(Reload-on-start method must be specified with ROS_LIB_METHOD=));
	   %return;
	%end;
	%if ( "&ROS_LIB_TABLEDEFAULT." eq "" ) %then %do;
	   %msg( type=ERROR,msg=%str(Reload-on-start table default must be specified with ROS_LIB_TABLEDEFAULT=));
	   %return;
	%end;

	%if ( "&OUT." eq "" ) %then %do;
		%let OUT=&DATA.;
	%end;

	%let ROS_LIB_METHOD=%UPCASE(&ROS_LIB_METHOD.);
	%let ROS_LIB_TABLEDEFAULT=%UPCASE(&ROS_LIB_TABLEDEFAULT.);

	data &OUT.;
	   set &DATA.;

	   %if ( "&ROS_LIB_METHOD." eq "ALL" ) %then %do;
	     if 1;
	   %end; 
	   %else %if ( "&ROS_LIB_METHOD." eq "SELECTIVE" ) %then %do;
	     if ( missing(ros_enabled) ) then ros_enabled="&ROS_LIB_TABLEDEFAULT";
		 if ros_enabled="YES";
	   %end;

	   if ( exist(fullref) );

	run;

%mend;

