#!/bin/sh

# ------------------------------------------------------------------------------
# Copyright (c) 2024 - 2025 by SAS Institute Inc., Cary, NC USA 27513
# ------------------------------------------------------------------------------
#
# Invoke Tomcat's catalina shell script to control the web application server.
# In addition, provide a server 'status' action based on the server's PID file
# and a corresponding system process.
#
# ------------------------------------------------------------------------------

ERROR_STATE=0
DEBUG_MODE=0


show_help()
{
   echo
   echo "Usage: tcruntime-ctl.sh <command>"
   echo "commands:"
   echo "   start       Start the web application server and wait for it to report ready"
   echo "   stop        Stop the web application server and wait for the process to end"
   echo "   status      Report the status of the web application server"
   echo "   restart     Stop the web application server and then restart it"
   echo "   configtest  Run a basic syntax check on server.xml - check exit code for result"
   echo "   version     What version of tomcat are you running?"
   echo
}


do_start()
{
   # Determine the size of the server's log file before startup.
   `ls $CATALINA_BASE/logs/server.log  >/dev/null 2>&1`
   if [ $? -eq 0 ]; then
      LOG_FILE_SIZE=`ls -l $CATALINA_BASE/logs/server.log | tr -s ' ' | cut -d ' ' -f5`
   fi
   if [ $DEBUG_MODE -ne 0 ]; then
      echo "DEBUG: LOG_FILE_SIZE = $LOG_FILE_SIZE"
   fi

   echo `date` - $(basename "$0") start >> "$CATALINA_BASE/logs/vfabric.execution.log"

   # Invoke the vendor shell script to start the server.
   $CATALINA_BASE/bin/catalina.sh start

   if [ $? -ne 0 ]; then
      echo "ERROR: Server start was not successful"
      ERROR_STATE=-1
   else
      # Once started, if a PID file was specified, wait for the
      # server to report that the startup process is complete.

      if [ -n "$CATALINA_PID" ]; then
          PID_OPTION=-Dcatalina.pid=$CATALINA_PID
      fi

      echo "NOTE: Waiting for server to report ready ..."
      "$JAVA_HOME/bin/java" -classpath $CLASSPATH \
                            -Dcatalina.home=$CATALINA_HOME \
                            -Dcatalina.base=$CATALINA_BASE \
                            $PID_OPTION \
                            -Dsas.vfabrictcsvr.execution.properties.file="$CATALINA_BASE/conf/vfabrictcsvr.properties" \
                            -Dsas.vfabrictcsvr.execution.log.size=$LOG_FILE_SIZE \
                            com.sas.vfabrictcsvr.execution.ExecutionInterface -wpr -verbose

      if [ $? -eq 0 ]; then
         echo "NOTE: Server start is complete and was successful"
      else
         echo "ERROR: Server start was not successful or cannot be verified as being complete"
         ERROR_STATE=-1
      fi
   fi
}


do_stop()
{
   # The wait-process-stop action will stop the server and then
   # wait for the process to terminate.

   if [ -n "$CATALINA_PID" ]; then
       PID_OPTION=-Dcatalina.pid=$CATALINA_PID
   fi

   echo `date` - $(basename "$0") stop >> "$CATALINA_BASE/logs/vfabric.execution.log"

   echo "NOTE: Waiting for server to terminate ..."
   "$JAVA_HOME/bin/java" -classpath $CLASSPATH \
                         -Dcatalina.home=$CATALINA_HOME \
                         -Dcatalina.base=$CATALINA_BASE \
                         $PID_OPTION \
                         -Dsas.vfabrictcsvr.execution.properties.file="$CATALINA_BASE/conf/vfabrictcsvr.properties" \
                         com.sas.vfabrictcsvr.execution.ExecutionInterface -wps -verbose
   if [ $? -eq 0 ]; then
      echo "NOTE: Server shutdown was successful"
   else
      echo "ERROR: Server shutdown cannot be verified as being successful"
      ERROR_STATE=-1
   fi
}


do_restart()
{
   echo `date` - $(basename "$0") restart >> "$CATALINA_BASE/logs/vfabric.execution.log"

   do_stop
   if [ $ERROR_STATE -eq 0 ]; then
      do_start
   else
      echo "WARNING: Not starting server because of error while stopping the server"
   fi
}


do_status()
{
   #
   # If a PID file exists and the process it specifies exists, then the server is running.
   #
   SASWEBAPPINSTANCE=`basename "$(cd "$CATALINA_BASE"; pwd -P)"`
   LOG_DIR=$(cd "$CATALINA_BASE/logs"; pwd -P)
   PID_FILE="$LOG_DIR/tcserver.pid"


   if [ -r "$PID_FILE" ]; then

      if [ -s "$PID_FILE" ]; then

         pid=`cat "$PID_FILE"`
         kill -0 $pid > /dev/null 2>&1
         if [ $? -eq 0 ]; then
            # Server is running
            echo "SAS Web App Server $SASWEBAPPINSTANCE is UP"
         else
            if [ $? -gt 0 ]; then
               echo "Found PID file $PID_FILE"
               echo "WARNING: There is no matching process ID: $pid"
            fi
         fi
      else
         echo "The PID file for $SASWEBAPPINSTANCE is empty"
      fi
   else
      echo "SAS Web App Server $SASWEBAPPINSTANCE is NOT UP"
   fi
}


do_configtest()
{
   $CATALINA_BASE/bin/catalina.sh configtest
}


do_version()
{
   "$JAVA_HOME/bin/java" \
   -classpath "$CATALINA_HOME/lib/catalina.jar" \
   org.apache.catalina.util.ServerInfo
}


bad_action()
{
   if [ "$1x" = "x" ]; then
      echo "ERROR: An action command is required"
   else
      echo "ERROR: Invalid action command:  $1"
   fi
   ERROR_STATE=1
   show_help
}


set_classpath()
{
   for JAR_FILE in $CATALINA_BASE/bin/*.jar; do
      CLASSPATH=$CLASSPATH:$JAR_FILE
   done
   for JAR_FILE in $CATALINA_BASE/lib/*.jar; do
      CLASSPATH=$CLASSPATH:$JAR_FILE
   done

   for JAR_FILE in $CATALINA_BASE/log4j2/lib/*.jar; do
      CLASSPATH=$CLASSPATH:$JAR_FILE
   done

   for JAR_FILE in $CATALINA_HOME/lib/*.jar; do
      CLASSPATH=$CLASSPATH:$JAR_FILE
   done

   CLASSPATH="$CLASSPATH:$CATALINA_HOME/bin/bootstrap.jar"
   CLASSPATH="$CLASSPATH:$CATALINA_BASE/log4j2/conf/log4j2.xml"

   if [ $DEBUG_MODE -ne 0 ]; then
      OS_NAME=`/bin/uname`
      echo "DEBUG: CLASSPATH ="
      if [ $OS_NAME == "AIX" ] || [ $OS_NAME == "SunOS" ]; then
         ORIGINAL_IFS=$IFS
         IFS=: 
         set -A array $CLASSPATH
         IFS=$ORIGINAL_IFS
         for PATHVAL in "${array[@]}"; do            
            echo "   - $PATHVAL" 
         done
      else
         set -- `echo $CLASSPATH | cut -d':' --output-delimiter=" " -f 1-` 
         for PATHVAL in "$@"; do 
            echo "   - $PATHVAL" 
         done
      fi
   fi
}



# ---------------------------------------------------------------------



for arg in "$@"; do
   shift
   if [ "$arg" = "-debug" ]; then
      DEBUG_MODE=1
      echo "DEBUG mode is enabled for the script only."
   else
      set -- "$@" "$arg"
   fi
done


if [ $DEBUG_MODE -ne 0 ]; then
   echo "DEBUG: Number of command line arguments = $#"
   COUNT=1
   for arg in "$@"; do
      echo "$COUNT = $arg"
      COUNT=`expr $COUNT + 1`
   done
fi


if [ -z "$CATALINA_HOME" ] || [ -z "$CATALINA_BASE" ] || [ -z "$JAVA_HOME" ]; then

   SRC_DIR=`dirname "$(cd "$(dirname "$0")"; pwd -P)/$(basename "$0")"`

   if [ -r $SRC_DIR/setenv.sh ]; then
       . "$SRC_DIR/setenv.sh"
   else
      echo "ERROR: File not found: $SRC_DIR/setenv.sh"
      ERROR_STATE=1
   fi
fi


if [ $ERROR_STATE -eq 0 ]; then

   if [ -z "$CATALINA_HOME" ]; then
      echo "ERROR: The CATALINA_HOME environment variable is not defined"
      ERROR_STATE=1
   else
      if [ $DEBUG_MODE -ne 0 ]; then
         echo "DEBUG: CATALINA_HOME = $CATALINA_HOME"
      fi
   fi


   if [ -z "$CATALINA_BASE" ]; then
      echo "ERROR: The CATALINA_BASE environment variable is not defined"
      ERROR_STATE=1
   else
      . "$CATALINA_BASE/bin/setenv.sh"
      if [ $DEBUG_MODE -ne 0 ]; then
         echo "DEBUG: CATALINA_BASE = $CATALINA_BASE"
      fi
   fi


   if [ -z "$JAVA_HOME" ]; then
      echo "ERROR: The JAVA_HOME environment variable is not defined"
      ERROR_STATE=1
   else
      if [ $DEBUG_MODE -ne 0 ]; then
         echo "DEBUG: JAVA_HOME = $JAVA_HOME"
         echo "DEBUG: Java Version:"
         "$JAVA_HOME/bin/java" -version
      fi
   fi
fi


if [ $ERROR_STATE -eq 0 ]; then

   if [ "$1" = "start" ] || [ "$1" = "stop" ] || [ "$1" = "restart" ]; then
      set_classpath
   fi

   case "$1" in
      start)      do_start       ;;
      stop)       do_stop        ;;
      restart)    do_restart     ;;
      status)     do_status      ;;
      configtest) do_configtest  ;;
      version)    do_version     ;;
      help)       show_help      ;;
      *)          bad_action     ;;
   esac
fi


exit $ERROR_STATE
