# ------------------------------------------------------------------------------
# Copyright (c) 2024 - 2025 by SAS Institute Inc., Cary, NC USA 27513
# ------------------------------------------------------------------------------
#
# PUROPSE:  Display registry values for a Windows service hosting a specified
#           Tomcat web application server.
#
# NOTES:    This utility retrieves parameters related to a Windows service from
#           the system registry using a hard coded root path value.  Should that
#           registry path change at some point in the future, a new path can be
#           specified in environment variable SWAS_PARAM_REG_PATH=
#
#           If defined, the value of SWAS_PARAM_REG_PATH= must be the full
#           registry path to the parameter keys, including the service name.
#           For the purpose of retrieving service parameters, the environment
#           variable overrides the -name argument.  The -name argument is still
#           required to retrieve additional values from a different registry path.
#
#           Registry paths with blanks must be wrapped in quotes when defining
#           the environment variable, but the value itself must not contain
#           quotes.  This can be accomplished by setting the value with quotes
#           around both the environment variable name and its value.
#
#           An example definition (broken across two lines for length):
#           > set "SWAS_PARAM_REG_PATH=HKLM:\SOFTWARE\WOW6432Node\Apache Software Foundation
#             \Procrun 2.0\SAS-SASConfig-Lev1-SASServer2_1-WebAppServer\Parameters"
#
# ------------------------------------------------------------------------------

<#
.SYNOPSIS
Display registry values for a Windows service hosting a SAS Web Application server.

.DESCRIPTION
The SAS Web Application servers can be executed as a Windows service, allowing
them to run without anyone having to login to the machine.  This utility is
used to display values from the Windows registry that define the server's run-time
environment and the parameters used to execute it.

A list of Windows service names for the SAS Web Application servers can be displayed
using the service control script SASServiceCtl.ps1.  Run the script with a -action
argument of "list" and a -name argument that contains a sub-string common to all of
the service names.  For example - SASServiceCtl.ps1 -action list -name SASServer

.PARAMETER Name
REQUIRED - The name of the Windows service hosting the SAS Web Application server.

.PARAMETER Debug
OPTIONAL - Output additional messages showing run-time parameters and values.

.PARAMETER Verbose
OPTIONAL - Output additional messages showing more in-depth information than would
normally be shown.

.EXAMPLE
SASServiceReg.ps1 -name SAS-SASConfig-Lev1-SASServer1_1-WebAppServer

#>

[ CmdletBinding ( PositionalBinding = $false ) ]

Param ( [Parameter ( Mandatory = $true,
                     HelpMessage = "(Required) Name of the service to operate on"
                   )
        ]
        [ValidateNotNullOrEmpty()]
        [String] $Name
      )


function ShowKey
{
   Param ( [string] $RegistryPath = $(throw "ERROR: A RegistryPath argument is mandatory.") )

   if ( (Test-Path -Path $RegistryPath ) -eq $false )
   {
      Write-Error -Message "ERROR: Registry path not found: $RegistryPath" `
                  -Category ObjectNotFound
   }

   else
   {
      Write-Output "`n${RegistryPath}: `n"
      $Key = Get-Item $RegistryPath
      $PropNames = $Key.GetValueNames()

      foreach ( $Name in $PropNames)
      {
         $PropValue = Get-ItemPropertyValue -Path $RegistryPath -Name $Name
         $PropType = $PropValue.GetType().Name
         if ( $PropType -eq 'String[]' )
         {
            Write-Output "   $Name = [List]"
            $PropValue | Where { $_  -ne "" } | Foreach-Object { "      $_" } | Format-List
         }

         else
         {
            Write-Output "   $Name = $PropValue"
         }
      }
   }
}


function Main()
{
   $ServicePath = "HKLM:\SYSTEM\CurrentControlSet\Services\$Name"
   $ParamPath = "HKLM:\SOFTWARE\WOW6432Node\Apache Software Foundation\Procrun 2.0\$Name\Parameters"

   if ( (Test-Path -Path $ServicePath ) -eq $false )
   {
      Write-Error -Message "ERROR: Windows service not defined: $Name" `
                  -Category ObjectNotFound
   }

   else
   {
      # FUTURE PROOFING: Should the registry path to the service parameters change at some
      # point in the future, a new path can be specified in an environment variable.
      $EnvVar = "SWAS_PARAM_REG_PATH"
      [String] $ParamPathOverride = [System.Environment]::GetEnvironmentVariable($EnvVar)
       if ( $ParamPathOverride.Length -gt 0 )
      {
         Write-Warning "Overriding service root parameter path with ${ParamPathOverride}"
         $ParamPath = $ParamPathOverride
      }

      write-output ""

      ShowKey ( $ServicePath )
      ShowKey ( $ParamPath )

      $ParamGroups = (Get-ChildItem -Path $ParamPath).psChildName
      foreach ( $GroupName in $ParamGroups )
      {
         ShowKey ( "$ParamPath\${GroupName}" )
      }
   }
}

#
# Invoke the main funciton for the script.
#
if ( $PSVersionTable.PSVersion.Major -lt 5 )
{
   Write-Error "This utility requires PowerShell version 5.0 or greater"
}

else
{
   Main
}

