/*
 * Decompiled with CFR 0.152.
 */
package org.apache.geode.internal.statistics;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.SimpleTimeZone;
import java.util.TimeZone;
import java.util.zip.GZIPInputStream;
import org.apache.geode.GemFireIOException;
import org.apache.geode.InternalGemFireException;
import org.apache.geode.internal.Assert;
import org.apache.geode.internal.ExitCode;
import org.apache.geode.internal.statistics.StatArchiveFormat;
import org.apache.geode.internal.statistics.StatArchiveWriter;

public class StatArchiveReader
implements StatArchiveFormat,
AutoCloseable {
    private final StatArchiveFile[] archives;
    private final boolean dump;
    private boolean closed = false;

    public StatArchiveReader(File[] archiveNames, ValueFilter[] filters, boolean autoClose) throws IOException {
        this.archives = new StatArchiveFile[archiveNames.length];
        this.dump = Boolean.getBoolean("StatArchiveReader.dumpall");
        for (int i = 0; i < archiveNames.length; ++i) {
            this.archives[i] = new StatArchiveFile(this, archiveNames[i], this.dump, filters);
        }
        this.update(false, autoClose);
        if (this.dump || Boolean.getBoolean("StatArchiveReader.dump")) {
            this.dump(new PrintWriter(System.out));
        }
    }

    public StatArchiveReader(String archiveName) throws IOException {
        this(new File[]{new File(archiveName)}, null, false);
    }

    public StatValue[] matchSpec(StatSpec spec) {
        StatArchiveFile[] archives;
        if (spec.getCombineType() == 2) {
            StatValue[] allValues = this.matchSpec(new RawStatSpec(spec));
            if (allValues.length == 0) {
                return allValues;
            }
            ComboValue cv = new ComboValue(allValues);
            return new StatValue[]{cv};
        }
        ArrayList l = new ArrayList();
        for (StatArchiveFile f : archives = this.getArchives()) {
            if (!spec.archiveMatches(f.getFile())) continue;
            f.matchSpec(spec, l);
        }
        StatValue[] result = new StatValue[l.size()];
        return l.toArray(result);
    }

    public boolean update() throws IOException {
        return this.update(true, false);
    }

    private boolean update(boolean doReset, boolean autoClose) throws IOException {
        StatArchiveFile[] archives;
        if (this.closed) {
            return false;
        }
        boolean result = false;
        for (StatArchiveFile f : archives = this.getArchives()) {
            if (f.update(doReset)) {
                result = true;
            }
            if (!autoClose) continue;
            f.close();
        }
        return result;
    }

    public List getResourceInstList() {
        return new ResourceInstList();
    }

    public StatArchiveFile[] getArchives() {
        return this.archives;
    }

    @Override
    public void close() throws IOException {
        if (!this.closed) {
            StatArchiveFile[] archives;
            for (StatArchiveFile f : archives = this.getArchives()) {
                f.close();
            }
            this.closed = true;
        }
    }

    private int getMemoryUsed() {
        StatArchiveFile[] archives;
        int result = 0;
        for (StatArchiveFile f : archives = this.getArchives()) {
            result += f.getMemoryUsed();
        }
        return result;
    }

    private void dump(PrintWriter stream) {
        StatArchiveFile[] archives;
        for (StatArchiveFile f : archives = this.getArchives()) {
            f.dump(stream);
        }
    }

    protected static double bitsToDouble(int type, long bits) {
        switch (type) {
            case 1: 
            case 2: 
            case 3: 
            case 4: 
            case 5: 
            case 6: 
            case 12: {
                return bits;
            }
            case 7: {
                return Float.intBitsToFloat((int)bits);
            }
            case 8: {
                return Double.longBitsToDouble(bits);
            }
        }
        throw new InternalGemFireException(String.format("Unexpected typecode %s", type));
    }

    private static void printStatValue(StatValue v, long startTime, long endTime, boolean nofilter, boolean persec, boolean persample, boolean prunezeros, boolean details) {
        v = v.createTrimmed(startTime, endTime);
        if (nofilter) {
            v.setFilter(0);
        } else if (persec) {
            v.setFilter(1);
        } else if (persample) {
            v.setFilter(2);
        }
        if (prunezeros && v.getSnapshotsMinimum() == 0.0 && v.getSnapshotsMaximum() == 0.0) {
            return;
        }
        System.out.println("  " + v.toString());
        if (details) {
            double[] snapshots;
            System.out.print("  values=");
            for (double snapshot : snapshots = v.getSnapshots()) {
                System.out.print(' ');
                System.out.print(snapshot);
            }
            System.out.println();
            String desc = v.getDescriptor().getDescription();
            if (desc != null && desc.length() > 0) {
                System.out.println("    " + desc);
            }
        }
    }

    public static void main(String[] args) throws IOException {
        StatArchiveReader reader;
        String archiveName = null;
        if (args.length > 1) {
            if (!args[0].equals("stat") || args.length > 3) {
                System.err.println("Usage: stat archiveName statType.statName");
                ExitCode.FATAL.doSystemExit();
            }
            archiveName = args[1];
            String statSpec = args[2];
            if (!statSpec.contains(".")) {
                throw new IllegalArgumentException("stat spec '" + statSpec + "' is malformed - use StatType.statName");
            }
            File archiveFile = new File(archiveName);
            if (!archiveFile.exists()) {
                throw new IllegalArgumentException("archive file does not exist: " + archiveName);
            }
            if (!archiveFile.canRead()) {
                throw new IllegalArgumentException("archive file exists but is unreadable: " + archiveName);
            }
            File[] archives = new File[]{archiveFile};
            ValueFilter[] filters = new SingleStatRawStatSpec[]{new SingleStatRawStatSpec(archiveName, args[2])};
            reader = new StatArchiveReader(archives, filters, false);
            StatValue[] statValues = reader.matchSpec((StatSpec)filters[0]);
            System.out.println(statSpec + " matched " + statValues.length + " stats...");
            for (StatValue value : statValues) {
                StatArchiveReader.printStatValue(value, -1L, -1L, true, false, false, false, true);
            }
            System.out.println();
            System.out.flush();
        } else {
            archiveName = args.length == 1 ? args[0] : "statArchive.gfs";
            reader = new StatArchiveReader(archiveName);
            System.out.println("DEBUG: memory used = " + reader.getMemoryUsed());
        }
        reader.close();
    }

    private static NumberFormat getNumberFormat() {
        NumberFormat nf = NumberFormat.getNumberInstance();
        nf.setMaximumFractionDigits(2);
        nf.setGroupingUsed(false);
        return nf;
    }

    public static class StatArchiveFile {
        private final StatArchiveReader reader;
        private InputStream is;
        private DataInputStream dataIn;
        private ValueFilter[] filters;
        private final File archiveName;
        private int archiveVersion;
        private ArchiveInfo info;
        private final boolean compressed;
        private final boolean updateOK;
        private final boolean dump;
        private boolean closed = false;
        protected int resourceInstSize = 0;
        protected ResourceInst[] resourceInstTable = null;
        private ResourceType[] resourceTypeTable = null;
        private final TimeStampSeries timeSeries = new TimeStampSeries();
        private final DateFormat timeFormatter = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss.SSS z");
        private static final int BUFFER_SIZE = 0x100000;
        private final ArrayList fileComboValues = new ArrayList();

        public StatArchiveFile(StatArchiveReader reader, File archiveName, boolean dump, ValueFilter[] filters) throws IOException {
            this.reader = reader;
            this.archiveName = archiveName;
            this.dump = dump;
            this.compressed = archiveName.getPath().endsWith(".gz");
            this.is = new FileInputStream(this.archiveName);
            this.dataIn = this.compressed ? new DataInputStream(new BufferedInputStream(new GZIPInputStream(this.is, 0x100000), 0x100000)) : new DataInputStream(new BufferedInputStream(this.is, 0x100000));
            this.updateOK = this.dataIn.markSupported();
            this.filters = this.createFilters(filters);
        }

        private ValueFilter[] createFilters(ValueFilter[] allFilters) {
            if (allFilters == null) {
                return new ValueFilter[0];
            }
            ArrayList<ValueFilter> l = new ArrayList<ValueFilter>();
            for (ValueFilter allFilter : allFilters) {
                if (!allFilter.archiveMatches(this.getFile())) continue;
                l.add(allFilter);
            }
            if (l.size() == allFilters.length) {
                return allFilters;
            }
            ValueFilter[] result = new ValueFilter[l.size()];
            return l.toArray(result);
        }

        StatArchiveReader getReader() {
            return this.reader;
        }

        void matchSpec(StatSpec spec, List matchedValues) {
            if (spec.getCombineType() == 1) {
                for (Object fileComboValue : this.fileComboValues) {
                    ResourceInst[] resources;
                    ComboValue v = (ComboValue)fileComboValue;
                    if (!spec.statMatches(v.getDescriptor().getName()) || !spec.typeMatches(v.getType().getName())) continue;
                    for (ResourceInst resource : resources = v.getResources()) {
                        if (spec.instanceMatches(resource.getName(), resource.getId())) continue;
                    }
                    matchedValues.add(v);
                    return;
                }
                ArrayList l = new ArrayList();
                this.matchSpec(new RawStatSpec(spec), l);
                if (l.size() != 0) {
                    ComboValue cv = new ComboValue(l);
                    this.fileComboValues.add(cv);
                    matchedValues.add(cv);
                }
            } else {
                for (int instIdx = 0; instIdx < this.resourceInstSize; ++instIdx) {
                    this.resourceInstTable[instIdx].matchSpec(spec, matchedValues);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public String formatTimeMillis(long ts) {
            DateFormat dateFormat = this.timeFormatter;
            synchronized (dateFormat) {
                return this.timeFormatter.format(new Date(ts));
            }
        }

        void setTimeZone(TimeZone z) {
            this.timeFormatter.setTimeZone(z);
        }

        TimeStampSeries getTimeStamps() {
            return this.timeSeries;
        }

        public boolean update(boolean doReset) throws IOException {
            if (this.closed) {
                return false;
            }
            if (!this.updateOK) {
                throw new InternalGemFireException("update of this type of file is not supported.");
            }
            if (doReset) {
                this.dataIn.reset();
            }
            int updateTokenCount = 0;
            while (this.readToken()) {
                ++updateTokenCount;
            }
            return updateTokenCount != 0;
        }

        public void dump(PrintWriter stream) {
            stream.print("archive=" + String.valueOf(this.archiveName));
            if (this.info != null) {
                this.info.dump(stream);
            }
            for (ResourceType resourceType : this.resourceTypeTable) {
                if (resourceType == null) continue;
                resourceType.dump(stream);
            }
            stream.print("time=");
            this.timeSeries.dump(stream);
            for (ResourceInst resourceInst : this.resourceInstTable) {
                if (resourceInst == null) continue;
                resourceInst.dump(stream);
            }
        }

        public File getFile() {
            return this.archiveName;
        }

        public void close() throws IOException {
            if (!this.closed) {
                this.closed = true;
                this.is.close();
                this.dataIn.close();
                this.is = null;
                this.dataIn = null;
                int typeCount = 0;
                if (this.resourceTypeTable != null) {
                    for (int i = 0; i < this.resourceTypeTable.length; ++i) {
                        if (this.resourceTypeTable[i] == null) continue;
                        if (this.resourceTypeTable[i].close()) {
                            this.resourceTypeTable[i] = null;
                            continue;
                        }
                        ++typeCount;
                    }
                    ResourceType[] newTypeTable = new ResourceType[typeCount];
                    typeCount = 0;
                    for (ResourceType resourceType : this.resourceTypeTable) {
                        if (resourceType == null) continue;
                        newTypeTable[typeCount] = resourceType;
                        ++typeCount;
                    }
                    this.resourceTypeTable = newTypeTable;
                }
                if (this.resourceInstTable != null) {
                    int instCount = 0;
                    for (int i = 0; i < this.resourceInstTable.length; ++i) {
                        if (this.resourceInstTable[i] == null) continue;
                        if (this.resourceInstTable[i].close()) {
                            this.resourceInstTable[i] = null;
                            continue;
                        }
                        ++instCount;
                    }
                    ResourceInst[] newInstTable = new ResourceInst[instCount];
                    instCount = 0;
                    for (ResourceInst resourceInst : this.resourceInstTable) {
                        if (resourceInst == null) continue;
                        newInstTable[instCount] = resourceInst;
                        ++instCount;
                    }
                    this.resourceInstTable = newInstTable;
                    this.resourceInstSize = instCount;
                }
                this.timeSeries.shrink();
                this.filters = null;
            }
        }

        public ArchiveInfo getArchiveInfo() {
            return this.info;
        }

        private void readHeaderToken() throws IOException {
            byte archiveVersion = this.dataIn.readByte();
            long startTimeStamp = this.dataIn.readLong();
            long systemId = this.dataIn.readLong();
            long systemStartTimeStamp = this.dataIn.readLong();
            int timeZoneOffset = this.dataIn.readInt();
            String timeZoneName = this.dataIn.readUTF();
            String systemDirectory = this.dataIn.readUTF();
            String productVersion = this.dataIn.readUTF();
            String os = this.dataIn.readUTF();
            String machine = this.dataIn.readUTF();
            if (archiveVersion <= 1) {
                throw new GemFireIOException(String.format("Archive version: %s is no longer supported.", archiveVersion), null);
            }
            if (archiveVersion > 4) {
                throw new GemFireIOException(String.format("Unsupported archive version: %s .  The supported version is: %s .", archiveVersion, (byte)4), null);
            }
            this.archiveVersion = archiveVersion;
            this.info = new ArchiveInfo(this, archiveVersion, startTimeStamp, systemStartTimeStamp, timeZoneOffset, timeZoneName, systemDirectory, systemId, productVersion, os, machine);
            this.resourceInstSize = 0;
            this.resourceInstTable = new ResourceInst[1024];
            this.resourceTypeTable = new ResourceType[256];
            this.timeSeries.setBase(startTimeStamp);
            if (this.dump) {
                this.info.dump(new PrintWriter(System.out));
            }
        }

        boolean loadType(String typeName) {
            if (this.filters == null || this.filters.length == 0) {
                return true;
            }
            for (ValueFilter filter : this.filters) {
                if (!filter.typeMatches(typeName)) continue;
                return true;
            }
            return false;
        }

        boolean loadStatDescriptor(StatDescriptor stat, ResourceType type) {
            if (!type.isLoaded()) {
                return false;
            }
            if (this.filters == null || this.filters.length == 0) {
                return true;
            }
            for (ValueFilter filter : this.filters) {
                if (!filter.statMatches(stat.getName()) || !filter.typeMatches(type.getName())) continue;
                return true;
            }
            stat.unload();
            return false;
        }

        boolean loadInstance(String textId, long numericId, ResourceType type) {
            if (!type.isLoaded()) {
                return false;
            }
            if (this.filters == null || this.filters.length == 0) {
                return true;
            }
            for (ValueFilter filter : this.filters) {
                StatDescriptor[] stats;
                if (!filter.typeMatches(type.getName()) || !filter.instanceMatches(textId, numericId)) continue;
                for (StatDescriptor stat : stats = type.getStats()) {
                    if (!stat.isLoaded() || !filter.statMatches(stat.getName())) continue;
                    return true;
                }
            }
            return false;
        }

        boolean loadStat(StatDescriptor stat, ResourceInst resource) {
            ResourceType type = resource.getType();
            if (!(resource.isLoaded() && type.isLoaded() && stat.isLoaded())) {
                return false;
            }
            if (this.filters == null || this.filters.length == 0) {
                return true;
            }
            String textId = resource.getName();
            long numericId = resource.getId();
            for (ValueFilter filter : this.filters) {
                if (!filter.statMatches(stat.getName()) || !filter.typeMatches(type.getName()) || !filter.instanceMatches(textId, numericId)) continue;
                return true;
            }
            return false;
        }

        private void readResourceTypeToken() throws IOException {
            ResourceType rt;
            int resourceTypeId = this.dataIn.readInt();
            String resourceTypeName = this.dataIn.readUTF();
            String resourceTypeDesc = this.dataIn.readUTF();
            int statCount = this.dataIn.readUnsignedShort();
            while (resourceTypeId >= this.resourceTypeTable.length) {
                ResourceType[] tmp = new ResourceType[this.resourceTypeTable.length + 128];
                System.arraycopy(this.resourceTypeTable, 0, tmp, 0, this.resourceTypeTable.length);
                this.resourceTypeTable = tmp;
            }
            Assert.assertTrue(this.resourceTypeTable[resourceTypeId] == null);
            if (this.loadType(resourceTypeName)) {
                rt = new ResourceType(resourceTypeId, resourceTypeName, resourceTypeDesc, statCount);
                if (this.dump) {
                    System.out.println("ResourceType id=" + resourceTypeId + " name=" + resourceTypeName + " statCount=" + statCount + " desc=" + resourceTypeDesc);
                }
            } else {
                rt = new ResourceType(resourceTypeId, resourceTypeName, statCount);
                if (this.dump) {
                    System.out.println("Not loading ResourceType id=" + resourceTypeId + " name=" + resourceTypeName);
                }
            }
            this.resourceTypeTable[resourceTypeId] = rt;
            for (int i = 0; i < statCount; ++i) {
                boolean isCounter;
                String statName = this.dataIn.readUTF();
                byte typeCode = this.dataIn.readByte();
                boolean largerBetter = isCounter = this.dataIn.readBoolean();
                if (this.archiveVersion >= 4) {
                    largerBetter = this.dataIn.readBoolean();
                }
                String units = this.dataIn.readUTF();
                String desc = this.dataIn.readUTF();
                rt.addStatDescriptor(this, i, statName, isCounter, largerBetter, typeCode, units, desc);
                if (!this.dump) continue;
                System.out.println("  " + i + "=" + statName + " isCtr=" + isCounter + " largerBetter=" + largerBetter + " typeCode=" + typeCode + " units=" + units + " desc=" + desc);
            }
        }

        private void readResourceInstanceCreateToken(boolean initialize) throws IOException {
            ResourceType type;
            int resourceInstId = this.dataIn.readInt();
            String name = this.dataIn.readUTF();
            long id = this.dataIn.readLong();
            int resourceTypeId = this.dataIn.readInt();
            while (resourceInstId >= this.resourceInstTable.length) {
                ResourceInst[] tmp = new ResourceInst[this.resourceInstTable.length + 128];
                System.arraycopy(this.resourceInstTable, 0, tmp, 0, this.resourceInstTable.length);
                this.resourceInstTable = tmp;
            }
            Assert.assertTrue(this.resourceInstTable[resourceInstId] == null);
            if (resourceInstId + 1 > this.resourceInstSize) {
                this.resourceInstSize = resourceInstId + 1;
            }
            if ((type = this.resourceTypeTable[resourceTypeId]) == null) {
                throw new IllegalStateException("ResourceType is missing for resourceTypeId " + resourceTypeId + ", resourceName " + name);
            }
            boolean loadInstance = this.loadInstance(name, id, this.resourceTypeTable[resourceTypeId]);
            this.resourceInstTable[resourceInstId] = new ResourceInst(this, resourceInstId, name, id, this.resourceTypeTable[resourceTypeId], loadInstance);
            if (this.dump) {
                System.out.println((loadInstance ? "Loaded" : "Did not load") + " resource instance " + resourceInstId);
                System.out.println("  name=" + name + " id=" + id + " typeId=" + resourceTypeId);
            }
            if (initialize) {
                StatDescriptor[] stats = this.resourceInstTable[resourceInstId].getType().getStats();
                for (int i = 0; i < stats.length; ++i) {
                    this.resourceInstTable[resourceInstId].initialValue(i, switch (stats[i].getTypeCode()) {
                        case 1 -> this.dataIn.readByte();
                        case 2, 3 -> this.dataIn.readByte();
                        case 12 -> this.dataIn.readUnsignedShort();
                        case 4 -> this.dataIn.readShort();
                        case 5, 6, 7, 8 -> this.readCompactValue();
                        default -> throw new IOException(String.format("unexpected typeCode value %s", stats[i].getTypeCode()));
                    });
                }
            }
        }

        private void readResourceInstanceDeleteToken() throws IOException {
            int resourceInstId = this.dataIn.readInt();
            Assert.assertTrue(this.resourceInstTable[resourceInstId] != null);
            this.resourceInstTable[resourceInstId].makeInactive();
            if (this.dump) {
                System.out.println("Delete resource instance " + resourceInstId);
            }
        }

        private int readResourceInstId() throws IOException {
            int token = this.dataIn.readUnsignedByte();
            if (token <= 252) {
                return token;
            }
            if (token == 255) {
                return -1;
            }
            if (token == 253) {
                return this.dataIn.readUnsignedShort();
            }
            return this.dataIn.readInt();
        }

        private int readTimeDelta() throws IOException {
            int result = this.dataIn.readUnsignedShort();
            if (result == 65535) {
                result = this.dataIn.readInt();
            }
            return result;
        }

        private long readCompactValue() throws IOException {
            return StatArchiveWriter.readCompactValue(this.dataIn);
        }

        private void readSampleToken() throws IOException {
            int millisSinceLastSample = this.readTimeDelta();
            if (this.dump) {
                System.out.println("ts=" + millisSinceLastSample);
            }
            int resourceInstId = this.readResourceInstId();
            while (resourceInstId != -1) {
                if (this.dump) {
                    System.out.print("  instId=" + resourceInstId);
                }
                StatDescriptor[] stats = this.resourceInstTable[resourceInstId].getType().getStats();
                int statOffset = this.dataIn.readUnsignedByte();
                while (statOffset != 255) {
                    long statDeltaBits = switch (stats[statOffset].getTypeCode()) {
                        case 1 -> this.dataIn.readByte();
                        case 2, 3 -> this.dataIn.readByte();
                        case 12 -> this.dataIn.readUnsignedShort();
                        case 4 -> this.dataIn.readShort();
                        case 5, 6, 7, 8 -> this.readCompactValue();
                        default -> throw new IOException(String.format("unexpected typeCode value %s", stats[statOffset].getTypeCode()));
                    };
                    if (this.resourceInstTable[resourceInstId].addValueSample(statOffset, statDeltaBits) && this.dump) {
                        System.out.print(" [" + statOffset + "]=" + statDeltaBits);
                    }
                    statOffset = this.dataIn.readUnsignedByte();
                }
                if (this.dump) {
                    System.out.println();
                }
                resourceInstId = this.readResourceInstId();
            }
            this.timeSeries.addTimeStamp(millisSinceLastSample);
            for (ResourceInst inst : this.resourceInstTable) {
                if (inst == null || !inst.isActive()) continue;
                inst.addTimeStamp();
            }
        }

        private boolean readToken() throws IOException {
            try {
                if (this.updateOK) {
                    this.dataIn.mark(0x100000);
                }
                byte token = this.dataIn.readByte();
                switch (token) {
                    case 77: {
                        this.readHeaderToken();
                        break;
                    }
                    case 1: {
                        this.readResourceTypeToken();
                        break;
                    }
                    case 2: {
                        this.readResourceInstanceCreateToken(false);
                        break;
                    }
                    case 4: {
                        this.readResourceInstanceCreateToken(true);
                        break;
                    }
                    case 3: {
                        this.readResourceInstanceDeleteToken();
                        break;
                    }
                    case 0: {
                        this.readSampleToken();
                        break;
                    }
                    default: {
                        throw new IOException(String.format("Unexpected token byte value: %s", token));
                    }
                }
                return true;
            }
            catch (EOFException ignore) {
                return false;
            }
        }

        protected int getMemoryUsed() {
            int result = 0;
            for (ResourceInst resourceInst : this.resourceInstTable) {
                if (resourceInst == null) continue;
                result += resourceInst.getMemoryUsed();
            }
            return result;
        }
    }

    public static interface ValueFilter {
        public boolean archiveMatches(File var1);

        public boolean typeMatches(String var1);

        public boolean statMatches(String var1);

        public boolean instanceMatches(String var1, long var2);
    }

    public static interface StatSpec
    extends ValueFilter {
        public static final int GLOBAL = 2;
        public static final int FILE = 1;
        public static final int NONE = 0;

        public int getCombineType();
    }

    private static class RawStatSpec
    implements StatSpec {
        private final StatSpec spec;

        RawStatSpec(StatSpec wrappedSpec) {
            this.spec = wrappedSpec;
        }

        @Override
        public int getCombineType() {
            return 0;
        }

        @Override
        public boolean typeMatches(String typeName) {
            return this.spec.typeMatches(typeName);
        }

        @Override
        public boolean statMatches(String statName) {
            return this.spec.statMatches(statName);
        }

        @Override
        public boolean instanceMatches(String textId, long numericId) {
            return this.spec.instanceMatches(textId, numericId);
        }

        @Override
        public boolean archiveMatches(File archive) {
            return this.spec.archiveMatches(archive);
        }
    }

    public static interface StatValue {
        public static final int FILTER_NONE = 0;
        public static final int FILTER_PERSEC = 1;
        public static final int FILTER_PERSAMPLE = 2;

        public StatValue createTrimmed(long var1, long var3);

        public boolean isTrimmedLeft();

        public ResourceType getType();

        public ResourceInst[] getResources();

        public long[] getRawAbsoluteTimeStamps();

        public long[] getRawAbsoluteTimeStampsWithSecondRes();

        public double[] getRawSnapshots();

        public double[] getSnapshots();

        public int getSnapshotsSize();

        public double getSnapshotsMinimum();

        public double getSnapshotsMaximum();

        public double getSnapshotsAverage();

        public double getSnapshotsStandardDeviation();

        public double getSnapshotsMostRecent();

        public boolean hasValueChanged();

        public int getFilter();

        public void setFilter(int var1);

        public StatDescriptor getDescriptor();
    }

    private static class ComboValue
    extends AbstractValue {
        private final ResourceType type;
        private final StatValue[] values;

        ComboValue(List valueList) {
            this(valueList.toArray(new StatValue[0]));
        }

        ComboValue(StatValue[] values) {
            this.values = values;
            this.filter = this.values[0].getFilter();
            String typeName = this.values[0].getType().getName();
            String statName = this.values[0].getDescriptor().getName();
            int bestTypeIdx = 0;
            for (int i = 1; i < this.values.length; ++i) {
                if (this.filter != this.values[i].getFilter()) {
                    throw new IllegalArgumentException("Cannot combine values with different filters.");
                }
                if (!typeName.equals(this.values[i].getType().getName())) {
                    throw new IllegalArgumentException("Cannot combine values with different types.");
                }
                if (!statName.equals(this.values[i].getDescriptor().getName())) {
                    throw new IllegalArgumentException("Cannot combine different stats.");
                }
                if (this.values[i].getDescriptor().isCounter()) {
                    if (!this.values[i].getDescriptor().isLargerBetter()) {
                        bestTypeIdx = i;
                        continue;
                    }
                    if (this.values[bestTypeIdx].getDescriptor().isCounter() != this.values[bestTypeIdx].getDescriptor().isLargerBetter()) continue;
                    bestTypeIdx = i;
                    continue;
                }
                if (!this.values[i].getDescriptor().isLargerBetter() || this.values[bestTypeIdx].getDescriptor().isCounter() != this.values[bestTypeIdx].getDescriptor().isLargerBetter()) continue;
                bestTypeIdx = i;
            }
            this.type = this.values[bestTypeIdx].getType();
            this.descriptor = this.values[bestTypeIdx].getDescriptor();
        }

        private ComboValue(ComboValue original, long startTime, long endTime) {
            this.startTime = startTime;
            this.endTime = endTime;
            this.type = original.getType();
            this.descriptor = original.getDescriptor();
            this.filter = original.getFilter();
            this.values = new StatValue[original.values.length];
            for (int i = 0; i < this.values.length; ++i) {
                this.values[i] = original.values[i].createTrimmed(startTime, endTime);
            }
        }

        @Override
        public StatValue createTrimmed(long startTime, long endTime) {
            if (startTime == this.startTime && endTime == this.endTime) {
                return this;
            }
            return new ComboValue(this, startTime, endTime);
        }

        @Override
        public ResourceType getType() {
            return this.type;
        }

        @Override
        public ResourceInst[] getResources() {
            HashSet<ResourceInst> set = new HashSet<ResourceInst>();
            for (StatValue value : this.values) {
                set.addAll(Arrays.asList(value.getResources()));
            }
            ResourceInst[] result = new ResourceInst[set.size()];
            return set.toArray(result);
        }

        @Override
        public boolean hasValueChanged() {
            return true;
        }

        public static boolean closeEnough(long v1, long v2, long delta) {
            return v1 == v2 || Math.abs(v1 - v2) / 2L <= delta;
        }

        public static boolean closer(long v, long prev, long next) {
            return Math.abs(v - prev) <= Math.abs(v - next);
        }

        private static boolean mustInsert(int nextIdx, long[] valueTimeStamps, long tsAtInsertPoint) {
            return nextIdx < valueTimeStamps.length && valueTimeStamps[nextIdx] <= tsAtInsertPoint;
        }

        @Override
        public long[] getRawAbsoluteTimeStampsWithSecondRes() {
            return this.getRawAbsoluteTimeStamps();
        }

        @Override
        public long[] getRawAbsoluteTimeStamps() {
            if (this.values.length == 0) {
                return new long[0];
            }
            long[] valueTimeStamps = this.values[0].getRawAbsoluteTimeStamps();
            int tsCount = valueTimeStamps.length + 1;
            long[] ourTimeStamps = new long[tsCount * 2 + 1];
            System.arraycopy(valueTimeStamps, 0, ourTimeStamps, 0, valueTimeStamps.length);
            ourTimeStamps[valueTimeStamps.length] = Long.MAX_VALUE;
            for (int i = 1; i < this.values.length; ++i) {
                valueTimeStamps = this.values[i].getRawAbsoluteTimeStamps();
                if (valueTimeStamps.length == 0) continue;
                int ourIdx = 0;
                int j = 0;
                long tsToInsert = valueTimeStamps[0] - 1000L;
                if (valueTimeStamps.length > 1) {
                    tsToInsert = valueTimeStamps[0] - (valueTimeStamps[1] - valueTimeStamps[0]);
                }
                while (j < valueTimeStamps.length) {
                    int endRunIdx;
                    long timeDelta = (valueTimeStamps[j] - tsToInsert) / 2L;
                    tsToInsert = valueTimeStamps[j];
                    long tsAtInsertPoint = ourTimeStamps[ourIdx];
                    while (tsToInsert > tsAtInsertPoint && !ComboValue.closeEnough(tsToInsert, tsAtInsertPoint, timeDelta)) {
                        tsAtInsertPoint = ourTimeStamps[++ourIdx];
                    }
                    if (ComboValue.closeEnough(tsToInsert, tsAtInsertPoint, timeDelta) && !ComboValue.mustInsert(j + 1, valueTimeStamps, tsAtInsertPoint)) {
                        ++j;
                        ++ourIdx;
                        while (!ComboValue.closer(tsToInsert, ourTimeStamps[ourIdx - 1], ourTimeStamps[ourIdx]) && !ComboValue.mustInsert(j, valueTimeStamps, ourTimeStamps[ourIdx])) {
                            ++ourIdx;
                        }
                        continue;
                    }
                    for (endRunIdx = j + 1; endRunIdx < valueTimeStamps.length && valueTimeStamps[endRunIdx] < tsAtInsertPoint && !ComboValue.closeEnough(valueTimeStamps[endRunIdx], tsAtInsertPoint, timeDelta); ++endRunIdx) {
                    }
                    int numToCopy = endRunIdx - j;
                    if (tsCount + numToCopy > ourTimeStamps.length) {
                        long[] tmp = new long[(tsCount + numToCopy) * 2];
                        System.arraycopy(ourTimeStamps, 0, tmp, 0, tsCount);
                        ourTimeStamps = tmp;
                    }
                    System.arraycopy(ourTimeStamps, ourIdx, ourTimeStamps, ourIdx + numToCopy, tsCount - ourIdx);
                    if (numToCopy == 1) {
                        ourTimeStamps[ourIdx] = valueTimeStamps[j];
                    } else {
                        System.arraycopy(valueTimeStamps, j, ourTimeStamps, ourIdx, numToCopy);
                    }
                    ourIdx += numToCopy;
                    tsCount += numToCopy;
                    j += numToCopy;
                }
            }
            int startIdx = 0;
            int endIdx = --tsCount - 1;
            if (this.startTime != -1L) {
                Assert.assertTrue(ourTimeStamps[startIdx] >= this.startTime);
            }
            if (this.endTime != -1L) {
                Assert.assertTrue(endIdx == startIdx - 1 || ourTimeStamps[endIdx] < this.endTime);
            }
            tsCount = endIdx - startIdx + 1;
            long[] tmp = new long[tsCount];
            System.arraycopy(ourTimeStamps, startIdx, tmp, 0, tsCount);
            ourTimeStamps = tmp;
            return ourTimeStamps;
        }

        @Override
        public double[] getRawSnapshots() {
            return this.getRawSnapshots(this.getRawAbsoluteTimeStamps());
        }

        private static boolean isClosest(long ts, long[] timeStamps, int curIdx) {
            if (curIdx >= timeStamps.length - 1) {
                return true;
            }
            if (ts == timeStamps[curIdx]) {
                return true;
            }
            return ComboValue.closer(ts, timeStamps[curIdx], timeStamps[curIdx + 1]);
        }

        @Override
        public boolean isTrimmedLeft() {
            for (StatValue value : this.values) {
                if (!value.isTrimmedLeft()) continue;
                return true;
            }
            return false;
        }

        private double[] getRawSnapshots(long[] ourTimeStamps) {
            double[] result = new double[ourTimeStamps.length];
            if (result.length > 0) {
                for (StatValue value : this.values) {
                    int j;
                    long[] valueTimeStamps = value.getRawAbsoluteTimeStamps();
                    double[] valueSnapshots = value.getRawSnapshots();
                    double currentValue = 0.0;
                    int curIdx = 0;
                    if (value.isTrimmedLeft() && valueSnapshots.length > 0) {
                        currentValue = valueSnapshots[0];
                    }
                    for (j = 0; j < valueSnapshots.length; ++j) {
                        while (!ComboValue.isClosest(valueTimeStamps[j], ourTimeStamps, curIdx)) {
                            if (this.descriptor.isCounter()) {
                                int n = curIdx;
                                result[n] = result[n] + currentValue;
                            }
                            ++curIdx;
                        }
                        if (curIdx >= result.length) {
                            int samplesSkipped = valueSnapshots.length - j;
                            StringBuilder msg = new StringBuilder(100);
                            msg.append("WARNING: dropping last ");
                            if (samplesSkipped == 1) {
                                msg.append("sample because it");
                            } else {
                                msg.append(samplesSkipped).append(" samples because they");
                            }
                            msg.append(" could not fit in the merged result.");
                            System.out.println(msg);
                            break;
                        }
                        currentValue = valueSnapshots[j];
                        int n = curIdx++;
                        result[n] = result[n] + currentValue;
                    }
                    if (!this.descriptor.isCounter()) continue;
                    j = curIdx;
                    while (j < result.length) {
                        int n = j++;
                        result[n] = result[n] + currentValue;
                    }
                }
            }
            return result;
        }

        @Override
        public double[] getSnapshots() {
            double[] result;
            if (this.filter != 0) {
                long[] timestamps = this.getRawAbsoluteTimeStamps();
                double[] snapshots = this.getRawSnapshots(timestamps);
                if (snapshots.length <= 1) {
                    return new double[0];
                }
                result = new double[snapshots.length - 1];
                for (int i = 0; i < result.length; ++i) {
                    double valueDelta = snapshots[i + 1] - snapshots[i];
                    if (this.filter == 1) {
                        long timeDelta = timestamps[i + 1] - timestamps[i];
                        result[i] = valueDelta / ((double)timeDelta / 1000.0);
                        continue;
                    }
                    result[i] = valueDelta;
                }
            } else {
                result = this.getRawSnapshots();
            }
            this.calcStats(result);
            return result;
        }
    }

    private class ResourceInstList
    extends AbstractList {
        protected ResourceInstList() {
        }

        @Override
        public Object get(int idx) {
            StatArchiveFile[] archives;
            int archiveIdx = 0;
            for (StatArchiveFile f : archives = StatArchiveReader.this.getArchives()) {
                if (idx < archiveIdx + f.resourceInstSize) {
                    return f.resourceInstTable[idx - archiveIdx];
                }
                archiveIdx += f.resourceInstSize;
            }
            return null;
        }

        @Override
        public int size() {
            StatArchiveFile[] archives;
            int result = 0;
            for (StatArchiveFile archive : archives = StatArchiveReader.this.getArchives()) {
                result += archive.resourceInstSize;
            }
            return result;
        }
    }

    public static class StatDescriptor {
        private boolean loaded = true;
        private String name;
        private final int offset;
        private final boolean isCounter;
        private final boolean largerBetter;
        private final byte typeCode;
        private String units;
        private String desc;

        protected void dump(PrintWriter stream) {
            stream.println("  " + this.name + ": type=" + this.typeCode + " offset=" + this.offset + (this.isCounter ? " counter" : "") + " units=" + this.units + " largerBetter=" + this.largerBetter + " desc=" + this.desc);
        }

        protected StatDescriptor(String name, int offset, boolean isCounter, boolean largerBetter, byte typeCode, String units, String desc) {
            this.name = name;
            this.offset = offset;
            this.isCounter = isCounter;
            this.largerBetter = largerBetter;
            this.typeCode = typeCode;
            this.units = units;
            this.desc = desc;
        }

        public boolean isLoaded() {
            return this.loaded;
        }

        void unload() {
            this.loaded = false;
            this.name = null;
            this.units = null;
            this.desc = null;
        }

        public byte getTypeCode() {
            return this.typeCode;
        }

        public String getName() {
            return this.name;
        }

        public boolean isCounter() {
            return this.isCounter;
        }

        public boolean isLargerBetter() {
            return this.largerBetter;
        }

        public String getUnits() {
            return this.units;
        }

        public String getDescription() {
            return this.desc;
        }

        public int getOffset() {
            return this.offset;
        }
    }

    private static class SingleStatRawStatSpec
    implements StatSpec {
        private final String archive;
        private final String statType;
        private final String statName;

        SingleStatRawStatSpec(String archive, String typeAndStat) {
            this.archive = archive;
            String[] parts = typeAndStat.split("\\.", 0);
            this.statType = parts[0];
            this.statName = parts[1];
        }

        @Override
        public boolean archiveMatches(File archive) {
            return true;
        }

        @Override
        public boolean typeMatches(String typeName) {
            return this.statType.equalsIgnoreCase(typeName);
        }

        @Override
        public boolean statMatches(String statName) {
            return this.statName.equalsIgnoreCase(statName);
        }

        @Override
        public boolean instanceMatches(String textId, long numericId) {
            return true;
        }

        @Override
        public int getCombineType() {
            return 0;
        }
    }

    public static class ResourceInst {
        private final boolean loaded;
        private final StatArchiveFile archive;
        private final ResourceType type;
        private final String name;
        private final long id;
        private boolean active = true;
        private final SimpleValue[] values;
        private int firstTSidx = -1;
        private int lastTSidx = -1;

        protected int getMemoryUsed() {
            int result = 0;
            if (this.values != null) {
                for (SimpleValue value : this.values) {
                    result += value.getMemoryUsed();
                }
            }
            return result;
        }

        public StatArchiveReader getReader() {
            return this.archive.getReader();
        }

        public String toString() {
            StringBuilder result = new StringBuilder();
            result.append(this.name).append(", ").append(this.id).append(", ").append(this.type.getName()).append(": \"").append(this.archive.formatTimeMillis(this.getFirstTimeMillis())).append('\"');
            if (!this.active) {
                result.append(" inactive");
            }
            result.append(" samples=" + this.getSampleCount());
            return result.toString();
        }

        public int getSampleCount() {
            if (this.active) {
                return this.archive.getTimeStamps().getSize() - this.firstTSidx;
            }
            return this.lastTSidx + 1 - this.firstTSidx;
        }

        public StatArchiveFile getArchive() {
            return this.archive;
        }

        protected void dump(PrintWriter stream) {
            stream.println(this.name + ": file=" + String.valueOf(this.getArchive().getFile()) + " id=" + this.id + (this.active ? "" : " deleted") + " start=" + this.archive.formatTimeMillis(this.getFirstTimeMillis()));
            for (SimpleValue value : this.values) {
                value.dump(stream);
            }
        }

        protected ResourceInst(StatArchiveFile archive, int uniqueId, String name, long id, ResourceType type, boolean loaded) {
            this.loaded = loaded;
            this.archive = archive;
            this.name = name;
            this.id = id;
            Assert.assertTrue(type != null);
            this.type = type;
            if (loaded) {
                StatDescriptor[] stats = type.getStats();
                this.values = new SimpleValue[stats.length];
                for (int i = 0; i < stats.length; ++i) {
                    this.values[i] = archive.loadStat(stats[i], this) ? new SimpleValue(this, stats[i]) : null;
                }
            } else {
                this.values = null;
            }
        }

        void matchSpec(StatSpec spec, List matchedValues) {
            if (spec.typeMatches(this.type.getName()) && spec.instanceMatches(this.getName(), this.getId())) {
                for (SimpleValue value : this.values) {
                    if (value == null || !spec.statMatches(value.getDescriptor().getName())) continue;
                    matchedValues.add(value);
                }
            }
        }

        protected void initialValue(int statOffset, long v) {
            if (this.values != null && this.values[statOffset] != null) {
                this.values[statOffset].initialValue(v);
            }
        }

        protected boolean addValueSample(int statOffset, long statDeltaBits) {
            if (this.values != null && this.values[statOffset] != null) {
                this.values[statOffset].prepareNextBits(statDeltaBits);
                return true;
            }
            return false;
        }

        public boolean isLoaded() {
            return this.loaded;
        }

        protected boolean close() {
            if (this.isLoaded()) {
                for (SimpleValue value : this.values) {
                    if (value == null) continue;
                    value.shrink();
                }
                return false;
            }
            return true;
        }

        protected int getFirstTimeStampIdx() {
            return this.firstTSidx;
        }

        protected long[] getAllRawTimeStamps() {
            return this.archive.getTimeStamps().getRawTimeStamps();
        }

        protected long getTimeBase() {
            return this.archive.getTimeStamps().getBase();
        }

        public double[] getSnapshotTimesMillis() {
            return this.archive.getTimeStamps().getTimeValuesSinceIdx(this.firstTSidx);
        }

        public StatValue[] getStatValues() {
            return this.values;
        }

        public StatValue getStatValue(String name) {
            SimpleValue result = null;
            StatDescriptor desc = this.getType().getStat(name);
            if (desc != null) {
                result = this.values[desc.getOffset()];
            }
            return result;
        }

        public String getName() {
            return this.name;
        }

        public long getId() {
            return this.id;
        }

        public long getFirstTimeMillis() {
            return this.archive.getTimeStamps().getMilliTimeStamp(this.firstTSidx);
        }

        public ResourceType getType() {
            return this.type;
        }

        protected void makeInactive() {
            this.active = false;
            this.lastTSidx = this.archive.getTimeStamps().getSize() - 1;
            this.close();
        }

        public boolean isActive() {
            return this.active;
        }

        protected void addTimeStamp() {
            if (this.loaded) {
                if (this.firstTSidx == -1) {
                    this.firstTSidx = this.archive.getTimeStamps().getSize() - 1;
                }
                for (SimpleValue value : this.values) {
                    if (value == null) continue;
                    value.addSample();
                }
            }
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + (int)(this.id ^ this.id >>> 32);
            result = 31 * result + (this.name == null ? 0 : this.name.hashCode());
            result = 31 * result + (this.type == null ? 0 : this.type.hashCode());
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            ResourceInst other = (ResourceInst)obj;
            if (this.id != other.id) {
                return false;
            }
            if (this.name == null ? other.name != null : !this.name.equals(other.name)) {
                return false;
            }
            if (this.type == null ? other.type != null : !this.type.equals(other.type)) {
                return false;
            }
            return this.firstTSidx == other.firstTSidx;
        }
    }

    public static class ArchiveInfo {
        private final StatArchiveFile archive;
        private final byte archiveVersion;
        private final long startTimeStamp;
        private final long systemStartTimeStamp;
        private final int timeZoneOffset;
        private final String timeZoneName;
        private final String systemDirectory;
        private final long systemId;
        private final String productVersion;
        private final String os;
        private final String machine;

        public ArchiveInfo(StatArchiveFile archive, byte archiveVersion, long startTimeStamp, long systemStartTimeStamp, int timeZoneOffset, String timeZoneName, String systemDirectory, long systemId, String productVersion, String os, String machine) {
            this.archive = archive;
            this.archiveVersion = archiveVersion;
            this.startTimeStamp = startTimeStamp;
            this.systemStartTimeStamp = systemStartTimeStamp;
            this.timeZoneOffset = timeZoneOffset;
            this.timeZoneName = timeZoneName;
            this.systemDirectory = systemDirectory;
            this.systemId = systemId;
            this.productVersion = productVersion;
            this.os = os;
            this.machine = machine;
            archive.setTimeZone(this.getTimeZone());
        }

        public long getStartTimeMillis() {
            return this.startTimeStamp;
        }

        public long getSystemStartTimeMillis() {
            return this.systemStartTimeStamp;
        }

        public long getSystemId() {
            return this.systemId;
        }

        public String getOs() {
            return this.os;
        }

        public String getMachine() {
            return this.machine;
        }

        public TimeZone getTimeZone() {
            TimeZone result = TimeZone.getTimeZone(this.timeZoneName);
            if (result.getRawOffset() != this.timeZoneOffset) {
                result = new SimpleTimeZone(this.timeZoneOffset, this.timeZoneName);
            }
            return result;
        }

        public String getProductVersion() {
            return this.productVersion;
        }

        public int getArchiveFormatVersion() {
            return this.archiveVersion;
        }

        public String getSystem() {
            return this.systemDirectory;
        }

        public String getArchiveFileName() {
            if (this.archive != null) {
                return this.archive.getFile().getPath();
            }
            return "";
        }

        public String toString() {
            StringWriter sw = new StringWriter();
            this.dump(new PrintWriter(sw));
            return sw.toString();
        }

        protected void dump(PrintWriter stream) {
            if (this.archive != null) {
                stream.println("archive=" + String.valueOf(this.archive.getFile()));
            }
            stream.println("archiveVersion=" + this.archiveVersion);
            if (this.archive != null) {
                stream.println("startDate=" + this.archive.formatTimeMillis(this.startTimeStamp));
            }
            stream.println("systemDirectory=" + this.systemDirectory);
            if (this.archive != null) {
                stream.println("systemStartDate=" + this.archive.formatTimeMillis(this.systemStartTimeStamp));
            }
            stream.println("systemId=" + this.systemId);
            stream.println("productVersion=" + this.productVersion);
            stream.println("osInfo=" + this.os);
            stream.println("machineInfo=" + this.machine);
        }
    }

    public static class ResourceType {
        private boolean loaded;
        private final String name;
        private String desc;
        private final StatDescriptor[] stats;
        private Map descriptorMap;

        public void dump(PrintWriter stream) {
            if (this.loaded) {
                stream.println(this.name + ": " + this.desc);
                for (StatDescriptor stat : this.stats) {
                    stat.dump(stream);
                }
            }
        }

        protected ResourceType(int id, String name, int statCount) {
            this.loaded = false;
            this.name = name;
            this.desc = null;
            this.stats = new StatDescriptor[statCount];
            this.descriptorMap = null;
        }

        protected ResourceType(int id, String name, String desc, int statCount) {
            this.loaded = true;
            this.name = name;
            this.desc = desc;
            this.stats = new StatDescriptor[statCount];
            this.descriptorMap = new HashMap();
        }

        public boolean isLoaded() {
            return this.loaded;
        }

        protected boolean close() {
            if (this.isLoaded()) {
                for (int i = 0; i < this.stats.length; ++i) {
                    if (this.stats[i] == null || this.stats[i].isLoaded()) continue;
                    this.stats[i] = null;
                }
                return false;
            }
            return true;
        }

        void unload() {
            this.loaded = false;
            this.desc = null;
            for (StatDescriptor stat : this.stats) {
                stat.unload();
            }
            this.descriptorMap.clear();
            this.descriptorMap = null;
        }

        protected void addStatDescriptor(StatArchiveFile archive, int offset, String name, boolean isCounter, boolean largerBetter, byte typeCode, String units, String desc) {
            StatDescriptor descriptor;
            this.stats[offset] = descriptor = new StatDescriptor(name, offset, isCounter, largerBetter, typeCode, units, desc);
            if (archive.loadStatDescriptor(descriptor, this)) {
                this.descriptorMap.put(name, descriptor);
            }
        }

        public String getName() {
            return this.name;
        }

        public StatDescriptor[] getStats() {
            return this.stats;
        }

        public StatDescriptor getStat(String name) {
            return (StatDescriptor)this.descriptorMap.get(name);
        }

        public String getDescription() {
            return this.desc;
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + (this.name == null ? 0 : this.name.hashCode());
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            ResourceType other = (ResourceType)obj;
            if (this.name == null) {
                return other.name == null;
            }
            return this.name.equals(other.name);
        }
    }

    private static class TimeStampSeries {
        private static final int GROW_SIZE = 256;
        int count = 0;
        long base = 0L;
        long[] timeStamps = new long[256];

        void dump(PrintWriter stream) {
            stream.print("[size=" + this.count);
            for (int i = 0; i < this.count; ++i) {
                if (i != 0) {
                    stream.print(", ");
                    stream.print(this.timeStamps[i] - this.timeStamps[i - 1]);
                    continue;
                }
                stream.print(" " + this.timeStamps[i]);
            }
            stream.println("]");
        }

        void shrink() {
            if (this.count < this.timeStamps.length) {
                long[] tmp = new long[this.count];
                System.arraycopy(this.timeStamps, 0, tmp, 0, this.count);
                this.timeStamps = tmp;
            }
        }

        TimeStampSeries() {
        }

        void setBase(long base) {
            this.base = base;
        }

        int getSize() {
            return this.count;
        }

        void addTimeStamp(int ts) {
            if (this.count >= this.timeStamps.length) {
                long[] tmp = new long[this.timeStamps.length + 256];
                System.arraycopy(this.timeStamps, 0, tmp, 0, this.timeStamps.length);
                this.timeStamps = tmp;
            }
            this.timeStamps[this.count] = this.count != 0 ? this.timeStamps[this.count - 1] + (long)ts : (long)ts;
            ++this.count;
        }

        long getBase() {
            return this.base;
        }

        long[] getRawTimeStamps() {
            return this.timeStamps;
        }

        long getMilliTimeStamp(int idx) {
            return this.base + this.timeStamps[idx];
        }

        double[] getTimeValuesSinceIdx(int idx) {
            int resultSize = this.count - idx;
            double[] result = new double[resultSize];
            for (int i = 0; i < resultSize; ++i) {
                result[i] = this.getMilliTimeStamp(idx + i);
            }
            return result;
        }
    }

    private static class BitSeries {
        int count = 0;
        long currentStartBits = 0L;
        long currentEndBits = 0L;
        long currentInterval = 0L;
        int currentCount = 0;
        int intervalIdx = -1;
        BitInterval[] intervals = null;

        protected int getMemoryUsed() {
            int result = 40;
            if (this.intervals != null) {
                result += 4 * this.intervals.length;
                for (int i = 0; i <= this.intervalIdx; ++i) {
                    result += this.intervals[i].getMemoryUsed();
                }
            }
            return result;
        }

        public double[] getValues(int typeCode) {
            return this.getValuesEx(typeCode, 0, this.getSize());
        }

        public double[] getValuesEx(int typeCode, int samplesToSkip, int resultSize) {
            double[] result = new double[resultSize];
            int idx = 0;
            for (int firstInterval = 0; samplesToSkip > 0 && firstInterval <= this.intervalIdx && this.intervals[firstInterval].getSampleCount() <= samplesToSkip; samplesToSkip -= this.intervals[firstInterval].getSampleCount(), ++firstInterval) {
            }
            for (int i = firstInterval; i <= this.intervalIdx; ++i) {
                idx += this.intervals[i].fill(result, idx, typeCode, samplesToSkip);
                samplesToSkip = 0;
            }
            if (this.currentCount != 0) {
                idx += BitInterval.create(this.currentStartBits, this.currentInterval, this.currentCount).fill(result, idx, typeCode, samplesToSkip);
            }
            if (idx != resultSize) {
                throw new InternalGemFireException(String.format("getValuesEx did not fill the last %s entries of its result.", resultSize - idx));
            }
            return result;
        }

        void dump(PrintWriter stream) {
            stream.print("[size=" + this.count + " intervals=" + (this.intervalIdx + 1) + " memused=" + this.getMemoryUsed() + " ");
            for (int i = 0; i <= this.intervalIdx; ++i) {
                if (i != 0) {
                    stream.print(", ");
                }
                this.intervals[i].dump(stream);
            }
            if (this.currentCount != 0) {
                if (this.intervalIdx != -1) {
                    stream.print(", ");
                }
                BitInterval.create(this.currentStartBits, this.currentInterval, this.currentCount).dump(stream);
            }
            stream.println("]");
        }

        BitSeries() {
        }

        void initialBits(long bits) {
            this.currentEndBits = bits;
        }

        int getSize() {
            return this.count;
        }

        void addBits(long deltaBits) {
            long bits = this.currentEndBits + deltaBits;
            if (this.currentCount == 0) {
                this.currentStartBits = bits;
                this.currentCount = 1;
            } else if (this.currentCount == 1) {
                this.currentInterval = deltaBits;
                ++this.currentCount;
            } else if (deltaBits == this.currentInterval) {
                ++this.currentCount;
            } else {
                if (this.intervalIdx == -1) {
                    this.intervals = new BitInterval[2];
                    this.intervalIdx = 0;
                    this.intervals[0] = BitInterval.create(this.currentStartBits, this.currentInterval, this.currentCount);
                } else if (!this.intervals[this.intervalIdx].attemptAdd(this.currentStartBits, this.currentInterval, this.currentCount)) {
                    ++this.intervalIdx;
                    if (this.intervalIdx >= this.intervals.length) {
                        BitInterval[] tmp = new BitInterval[this.intervals.length * 2];
                        System.arraycopy(this.intervals, 0, tmp, 0, this.intervals.length);
                        this.intervals = tmp;
                    }
                    this.intervals[this.intervalIdx] = BitInterval.create(this.currentStartBits, this.currentInterval, this.currentCount);
                }
                this.currentStartBits = bits;
                this.currentCount = 1;
            }
            this.currentEndBits = bits;
            ++this.count;
        }

        void shrink() {
            int currentSize;
            if (this.intervals != null && (currentSize = this.intervalIdx + 1) < this.intervals.length) {
                BitInterval[] tmp = new BitInterval[currentSize];
                System.arraycopy(this.intervals, 0, tmp, 0, currentSize);
                this.intervals = tmp;
            }
        }
    }

    private static class BitExplicitLongInterval
    extends BitInterval {
        long[] bitArray = null;

        @Override
        int getMemoryUsed() {
            int result = super.getMemoryUsed() + 4 + 4;
            if (this.bitArray != null) {
                result += this.bitArray.length * 8;
            }
            return result;
        }

        @Override
        int fill(double[] values, int valueOffset, int typeCode, int skipCount) {
            int fillcount = values.length - valueOffset;
            int maxCount = this.count - skipCount;
            if (fillcount > maxCount) {
                fillcount = maxCount;
            }
            for (int i = 0; i < fillcount; ++i) {
                values[valueOffset + i] = StatArchiveReader.bitsToDouble(typeCode, this.bitArray[skipCount + i]);
            }
            return fillcount;
        }

        @Override
        void dump(PrintWriter stream) {
            stream.print("(count=" + this.count + " ");
            for (int i = 0; i < this.count; ++i) {
                if (i != 0) {
                    stream.print(", ");
                }
                stream.print(this.bitArray[i]);
            }
            stream.print(")");
        }

        BitExplicitLongInterval(long bits, long interval, int addCount) {
            this.count = addCount;
            this.bitArray = new long[this.count * 2];
            for (int i = 0; i < this.count; ++i) {
                this.bitArray[i] = bits;
                bits += interval;
            }
        }

        @Override
        boolean attemptAdd(long addBits, long addInterval, int addCount) {
            if (addCount <= 3) {
                if (this.count + addCount >= this.bitArray.length) {
                    long[] tmp = new long[(this.count + addCount) * 2];
                    System.arraycopy(this.bitArray, 0, tmp, 0, this.bitArray.length);
                    this.bitArray = tmp;
                }
                for (int i = 0; i < addCount; ++i) {
                    this.bitArray[this.count++] = addBits;
                    addBits += addInterval;
                }
                return true;
            }
            return false;
        }
    }

    private static class BitExplicitIntInterval
    extends BitInterval {
        long firstValue;
        long lastValue;
        int[] bitIntervals = null;

        @Override
        int getMemoryUsed() {
            int result = super.getMemoryUsed() + 4 + 8 + 8 + 4;
            if (this.bitIntervals != null) {
                result += this.bitIntervals.length * 4;
            }
            return result;
        }

        @Override
        int fill(double[] values, int valueOffset, int typeCode, int skipCount) {
            int i;
            int fillcount = values.length - valueOffset;
            int maxCount = this.count - skipCount;
            if (fillcount > maxCount) {
                fillcount = maxCount;
            }
            long bitValue = this.firstValue;
            for (i = 0; i < skipCount; ++i) {
                bitValue += (long)this.bitIntervals[i];
            }
            for (i = 0; i < fillcount; ++i) {
                values[valueOffset + i] = StatArchiveReader.bitsToDouble(typeCode, bitValue += (long)this.bitIntervals[skipCount + i]);
            }
            return fillcount;
        }

        @Override
        void dump(PrintWriter stream) {
            stream.print("(intIntervalCount=" + this.count + " start=" + this.firstValue);
            for (int i = 0; i < this.count; ++i) {
                if (i != 0) {
                    stream.print(", ");
                }
                stream.print(this.bitIntervals[i]);
            }
            stream.print(")");
        }

        BitExplicitIntInterval(long bits, long interval, int addCount) {
            this.count = addCount;
            this.firstValue = bits;
            this.lastValue = bits + interval * (long)(addCount - 1);
            this.bitIntervals = new int[this.count * 2];
            this.bitIntervals[0] = 0;
            for (int i = 1; i < this.count; ++i) {
                this.bitIntervals[i] = (int)interval;
            }
        }

        @Override
        boolean attemptAdd(long addBits, long addInterval, int addCount) {
            long firstInterval;
            if (addCount <= 4 && addInterval <= Integer.MAX_VALUE && addInterval >= Integer.MIN_VALUE && (firstInterval = addBits - this.lastValue) <= Integer.MAX_VALUE && firstInterval >= Integer.MIN_VALUE) {
                this.lastValue = addBits + addInterval * (long)(addCount - 1);
                if (this.count + addCount >= this.bitIntervals.length) {
                    int[] tmp = new int[(this.count + addCount) * 2];
                    System.arraycopy(this.bitIntervals, 0, tmp, 0, this.bitIntervals.length);
                    this.bitIntervals = tmp;
                }
                this.bitIntervals[this.count++] = (int)firstInterval;
                for (int i = 1; i < addCount; ++i) {
                    this.bitIntervals[this.count++] = (int)addInterval;
                }
                return true;
            }
            return false;
        }
    }

    private static class BitExplicitShortInterval
    extends BitInterval {
        long firstValue;
        long lastValue;
        short[] bitIntervals = null;

        @Override
        int getMemoryUsed() {
            int result = super.getMemoryUsed() + 4 + 8 + 8 + 4;
            if (this.bitIntervals != null) {
                result += this.bitIntervals.length * 2;
            }
            return result;
        }

        @Override
        int fill(double[] values, int valueOffset, int typeCode, int skipCount) {
            int i;
            int fillcount = values.length - valueOffset;
            int maxCount = this.count - skipCount;
            if (fillcount > maxCount) {
                fillcount = maxCount;
            }
            long bitValue = this.firstValue;
            for (i = 0; i < skipCount; ++i) {
                bitValue += (long)this.bitIntervals[i];
            }
            for (i = 0; i < fillcount; ++i) {
                values[valueOffset + i] = StatArchiveReader.bitsToDouble(typeCode, bitValue += (long)this.bitIntervals[skipCount + i]);
            }
            return fillcount;
        }

        @Override
        void dump(PrintWriter stream) {
            stream.print("(shortIntervalCount=" + this.count + " start=" + this.firstValue);
            for (int i = 0; i < this.count; ++i) {
                if (i != 0) {
                    stream.print(", ");
                }
                stream.print(this.bitIntervals[i]);
            }
            stream.print(")");
        }

        BitExplicitShortInterval(long bits, long interval, int addCount) {
            this.count = addCount;
            this.firstValue = bits;
            this.lastValue = bits + interval * (long)(addCount - 1);
            this.bitIntervals = new short[this.count * 2];
            this.bitIntervals[0] = 0;
            for (int i = 1; i < this.count; ++i) {
                this.bitIntervals[i] = (short)interval;
            }
        }

        @Override
        boolean attemptAdd(long addBits, long addInterval, int addCount) {
            long firstInterval;
            if (addCount <= 6 && addInterval <= 32767L && addInterval >= -32768L && (firstInterval = addBits - this.lastValue) <= 32767L && firstInterval >= -32768L) {
                this.lastValue = addBits + addInterval * (long)(addCount - 1);
                if (this.count + addCount >= this.bitIntervals.length) {
                    short[] tmp = new short[(this.count + addCount) * 2];
                    System.arraycopy(this.bitIntervals, 0, tmp, 0, this.bitIntervals.length);
                    this.bitIntervals = tmp;
                }
                this.bitIntervals[this.count++] = (short)firstInterval;
                for (int i = 1; i < addCount; ++i) {
                    this.bitIntervals[this.count++] = (short)addInterval;
                }
                return true;
            }
            return false;
        }
    }

    private static class BitExplicitByteInterval
    extends BitInterval {
        long firstValue;
        long lastValue;
        byte[] bitIntervals = null;

        @Override
        int getMemoryUsed() {
            int result = super.getMemoryUsed() + 4 + 8 + 8 + 4;
            if (this.bitIntervals != null) {
                result += this.bitIntervals.length;
            }
            return result;
        }

        @Override
        int fill(double[] values, int valueOffset, int typeCode, int skipCount) {
            int i;
            int fillcount = values.length - valueOffset;
            int maxCount = this.count - skipCount;
            if (fillcount > maxCount) {
                fillcount = maxCount;
            }
            long bitValue = this.firstValue;
            for (i = 0; i < skipCount; ++i) {
                bitValue += (long)this.bitIntervals[i];
            }
            for (i = 0; i < fillcount; ++i) {
                values[valueOffset + i] = StatArchiveReader.bitsToDouble(typeCode, bitValue += (long)this.bitIntervals[skipCount + i]);
            }
            return fillcount;
        }

        @Override
        void dump(PrintWriter stream) {
            stream.print("(byteIntervalCount=" + this.count + " start=" + this.firstValue);
            for (int i = 0; i < this.count; ++i) {
                if (i != 0) {
                    stream.print(", ");
                }
                stream.print(this.bitIntervals[i]);
            }
            stream.print(")");
        }

        BitExplicitByteInterval(long bits, long interval, int addCount) {
            this.count = addCount;
            this.firstValue = bits;
            this.lastValue = bits + interval * (long)(addCount - 1);
            this.bitIntervals = new byte[this.count * 2];
            this.bitIntervals[0] = 0;
            for (int i = 1; i < this.count; ++i) {
                this.bitIntervals[i] = (byte)interval;
            }
        }

        @Override
        boolean attemptAdd(long addBits, long addInterval, int addCount) {
            long firstInterval;
            if (addCount <= 11 && addInterval <= 127L && addInterval >= -128L && (firstInterval = addBits - this.lastValue) <= 127L && firstInterval >= -128L) {
                this.lastValue = addBits + addInterval * (long)(addCount - 1);
                if (this.count + addCount >= this.bitIntervals.length) {
                    byte[] tmp = new byte[(this.count + addCount) * 2];
                    System.arraycopy(this.bitIntervals, 0, tmp, 0, this.bitIntervals.length);
                    this.bitIntervals = tmp;
                }
                this.bitIntervals[this.count++] = (byte)firstInterval;
                for (int i = 1; i < addCount; ++i) {
                    this.bitIntervals[this.count++] = (byte)addInterval;
                }
                return true;
            }
            return false;
        }
    }

    private static class BitZeroLongInterval
    extends BitZeroInterval {
        long bits;

        @Override
        int getMemoryUsed() {
            return super.getMemoryUsed() + 8;
        }

        @Override
        long getBits() {
            return this.bits;
        }

        BitZeroLongInterval(long bits, int count) {
            super(count);
            this.bits = bits;
        }
    }

    private static class BitZeroIntInterval
    extends BitZeroInterval {
        int bits;

        @Override
        int getMemoryUsed() {
            return super.getMemoryUsed() + 4;
        }

        @Override
        long getBits() {
            return this.bits;
        }

        BitZeroIntInterval(int bits, int count) {
            super(count);
            this.bits = bits;
        }
    }

    private static abstract class BitZeroInterval
    extends BitInterval {
        @Override
        int getMemoryUsed() {
            return super.getMemoryUsed() + 4;
        }

        abstract long getBits();

        @Override
        int fill(double[] values, int valueOffset, int typeCode, int skipCount) {
            int fillcount = values.length - valueOffset;
            int maxCount = this.count - skipCount;
            if (fillcount > maxCount) {
                fillcount = maxCount;
            }
            double value = StatArchiveReader.bitsToDouble(typeCode, this.getBits());
            for (int i = 0; i < fillcount; ++i) {
                values[valueOffset + i] = value;
            }
            return fillcount;
        }

        @Override
        void dump(PrintWriter stream) {
            stream.print(this.getBits());
            if (this.count > 1) {
                stream.print("r" + this.count);
            }
        }

        BitZeroInterval(int count) {
            this.count = count;
        }

        @Override
        boolean attemptAdd(long addBits, long addInterval, int addCount) {
            if (addInterval == 0L && addBits == this.getBits()) {
                this.count += addCount;
                return true;
            }
            return false;
        }
    }

    private static class BitNonZeroLongLongInterval
    extends BitNonZeroInterval {
        long bits;
        long interval;

        @Override
        int getMemoryUsed() {
            return super.getMemoryUsed() + 16;
        }

        @Override
        long getBits() {
            return this.bits;
        }

        @Override
        long getInterval() {
            return this.interval;
        }

        BitNonZeroLongLongInterval(long bits, long interval, int count) {
            super(count);
            this.bits = bits;
            this.interval = interval;
        }
    }

    private static class BitNonZeroLongIntInterval
    extends BitNonZeroInterval {
        long bits;
        int interval;

        @Override
        int getMemoryUsed() {
            return super.getMemoryUsed() + 12;
        }

        @Override
        long getBits() {
            return this.bits;
        }

        @Override
        long getInterval() {
            return this.interval;
        }

        BitNonZeroLongIntInterval(long bits, int interval, int count) {
            super(count);
            this.bits = bits;
            this.interval = interval;
        }
    }

    private static class BitNonZeroIntLongInterval
    extends BitNonZeroInterval {
        int bits;
        long interval;

        @Override
        int getMemoryUsed() {
            return super.getMemoryUsed() + 12;
        }

        @Override
        long getBits() {
            return this.bits;
        }

        @Override
        long getInterval() {
            return this.interval;
        }

        BitNonZeroIntLongInterval(int bits, long interval, int count) {
            super(count);
            this.bits = bits;
            this.interval = interval;
        }
    }

    private static class BitNonZeroIntIntInterval
    extends BitNonZeroInterval {
        int bits;
        int interval;

        @Override
        int getMemoryUsed() {
            return super.getMemoryUsed() + 8;
        }

        @Override
        long getBits() {
            return this.bits;
        }

        @Override
        long getInterval() {
            return this.interval;
        }

        BitNonZeroIntIntInterval(int bits, int interval, int count) {
            super(count);
            this.bits = bits;
            this.interval = interval;
        }
    }

    private static abstract class BitNonZeroInterval
    extends BitInterval {
        @Override
        int getMemoryUsed() {
            return super.getMemoryUsed() + 4;
        }

        abstract long getBits();

        abstract long getInterval();

        @Override
        int fill(double[] values, int valueOffset, int typeCode, int skipCount) {
            int fillcount = values.length - valueOffset;
            int maxCount = this.count - skipCount;
            if (fillcount > maxCount) {
                fillcount = maxCount;
            }
            long base = this.getBits();
            long interval = this.getInterval();
            base += (long)skipCount * interval;
            for (int i = 0; i < fillcount; ++i) {
                values[valueOffset + i] = StatArchiveReader.bitsToDouble(typeCode, base);
                base += interval;
            }
            return fillcount;
        }

        @Override
        void dump(PrintWriter stream) {
            stream.print(this.getBits());
            if (this.count > 1) {
                long interval = this.getInterval();
                if (interval != 0L) {
                    stream.print("+=" + interval);
                }
                stream.print("r" + this.count);
            }
        }

        BitNonZeroInterval(int count) {
            this.count = count;
        }

        @Override
        boolean attemptAdd(long addBits, long addInterval, int addCount) {
            if (addInterval == this.getInterval() && addBits == this.getBits() + addInterval * (long)(this.count - 1)) {
                this.count += addCount;
                return true;
            }
            return false;
        }
    }

    private static abstract class BitInterval {
        protected int count;

        private BitInterval() {
        }

        abstract int fill(double[] var1, int var2, int var3, int var4);

        abstract void dump(PrintWriter var1);

        abstract boolean attemptAdd(long var1, long var3, int var5);

        int getMemoryUsed() {
            return 0;
        }

        public int getSampleCount() {
            return this.count;
        }

        static BitInterval create(long bits, long interval, int count) {
            if (interval == 0L) {
                if (bits <= Integer.MAX_VALUE && bits >= Integer.MIN_VALUE) {
                    return new BitZeroIntInterval((int)bits, count);
                }
                return new BitZeroLongInterval(bits, count);
            }
            if (count <= 3) {
                if (interval <= 127L && interval >= -128L) {
                    return new BitExplicitByteInterval(bits, interval, count);
                }
                if (interval <= 32767L && interval >= -32768L) {
                    return new BitExplicitShortInterval(bits, interval, count);
                }
                if (interval <= Integer.MAX_VALUE && interval >= Integer.MIN_VALUE) {
                    return new BitExplicitIntInterval(bits, interval, count);
                }
                return new BitExplicitLongInterval(bits, interval, count);
            }
            boolean smallBits = false;
            boolean smallInterval = false;
            if (bits <= Integer.MAX_VALUE && bits >= Integer.MIN_VALUE) {
                smallBits = true;
            }
            if (interval <= Integer.MAX_VALUE && interval >= Integer.MIN_VALUE) {
                smallInterval = true;
            }
            if (smallBits) {
                if (smallInterval) {
                    return new BitNonZeroIntIntInterval((int)bits, (int)interval, count);
                }
                return new BitNonZeroIntLongInterval((int)bits, interval, count);
            }
            if (smallInterval) {
                return new BitNonZeroLongIntInterval(bits, (int)interval, count);
            }
            return new BitNonZeroLongLongInterval(bits, interval, count);
        }
    }

    private static class SimpleValue
    extends AbstractValue {
        private final ResourceInst resource;
        private boolean useNextBits = false;
        private long nextBits;
        private final BitSeries series;
        private boolean valueChangeNoticed = false;

        @Override
        public StatValue createTrimmed(long startTime, long endTime) {
            if (startTime == this.startTime && endTime == this.endTime) {
                return this;
            }
            return new SimpleValue(this, startTime, endTime);
        }

        protected SimpleValue(ResourceInst resource, StatDescriptor sd) {
            this.resource = resource;
            this.filter = sd.isCounter() ? 1 : 0;
            this.descriptor = sd;
            this.series = new BitSeries();
            this.statsValid = false;
        }

        private SimpleValue(SimpleValue in, long startTime, long endTime) {
            this.startTime = startTime;
            this.endTime = endTime;
            this.useNextBits = in.useNextBits;
            this.nextBits = in.nextBits;
            this.resource = in.resource;
            this.series = in.series;
            this.descriptor = in.descriptor;
            this.filter = in.filter;
            this.statsValid = false;
            this.valueChangeNoticed = true;
        }

        @Override
        public ResourceType getType() {
            return this.resource.getType();
        }

        @Override
        public ResourceInst[] getResources() {
            return new ResourceInst[]{this.resource};
        }

        @Override
        public boolean isTrimmedLeft() {
            return this.getStartIdx() != 0;
        }

        private int getStartIdx() {
            int startIdx = 0;
            if (this.startTime != -1L) {
                long startTimeStamp = this.startTime - this.resource.getTimeBase();
                long[] timestamps = this.resource.getAllRawTimeStamps();
                for (int i = this.resource.getFirstTimeStampIdx(); i < this.resource.getFirstTimeStampIdx() + this.series.getSize() && timestamps[i] < startTimeStamp; ++i) {
                    ++startIdx;
                }
            }
            return startIdx;
        }

        private int getEndIdx(int startIdx) {
            int endIdx = this.series.getSize() - 1;
            if (this.endTime != -1L) {
                long endTimeStamp = this.endTime - this.resource.getTimeBase();
                long[] timestamps = this.resource.getAllRawTimeStamps();
                endIdx = startIdx - 1;
                for (int i = this.resource.getFirstTimeStampIdx() + startIdx; i < this.resource.getFirstTimeStampIdx() + this.series.getSize() && timestamps[i] < endTimeStamp; ++i) {
                    ++endIdx;
                }
                Assert.assertTrue(endIdx == startIdx - 1 || timestamps[endIdx] < endTimeStamp);
            }
            return endIdx;
        }

        @Override
        public double[] getSnapshots() {
            double[] result;
            int startIdx = this.getStartIdx();
            int endIdx = this.getEndIdx(startIdx);
            int resultSize = endIdx - startIdx + 1;
            if (this.filter != 0 && resultSize > 1) {
                long[] timestamps = null;
                if (this.filter == 1) {
                    timestamps = this.resource.getAllRawTimeStamps();
                }
                result = new double[resultSize - 1];
                int tsIdx = this.resource.getFirstTimeStampIdx() + startIdx;
                double[] values = this.series.getValuesEx(this.descriptor.getTypeCode(), startIdx, resultSize);
                for (int i = 0; i < result.length; ++i) {
                    double valueDelta = values[i + 1] - values[i];
                    if (this.filter == 1) {
                        double timeDelta = timestamps[tsIdx + i + 1] - timestamps[tsIdx + i];
                        valueDelta /= timeDelta / 1000.0;
                    }
                    result[i] = valueDelta;
                }
            } else {
                result = this.series.getValuesEx(this.descriptor.getTypeCode(), startIdx, resultSize);
            }
            this.calcStats(result);
            return result;
        }

        @Override
        public double[] getRawSnapshots() {
            int startIdx = this.getStartIdx();
            int endIdx = this.getEndIdx(startIdx);
            int resultSize = endIdx - startIdx + 1;
            return this.series.getValuesEx(this.descriptor.getTypeCode(), startIdx, resultSize);
        }

        @Override
        public long[] getRawAbsoluteTimeStampsWithSecondRes() {
            long[] result = this.getRawAbsoluteTimeStamps();
            int i = 0;
            while (i < result.length) {
                int n = i;
                result[n] = result[n] + 500L;
                int n2 = i;
                result[n2] = result[n2] / 1000L;
                int n3 = i++;
                result[n3] = result[n3] * 1000L;
            }
            return result;
        }

        @Override
        public long[] getRawAbsoluteTimeStamps() {
            int startIdx = this.getStartIdx();
            int endIdx = this.getEndIdx(startIdx);
            int resultSize = endIdx - startIdx + 1;
            if (resultSize <= 0) {
                return new long[0];
            }
            long[] result = new long[resultSize];
            long[] timestamps = this.resource.getAllRawTimeStamps();
            int tsIdx = this.resource.getFirstTimeStampIdx() + startIdx;
            long base = this.resource.getTimeBase();
            for (int i = 0; i < resultSize; ++i) {
                result[i] = base + timestamps[tsIdx + i];
            }
            return result;
        }

        @Override
        public boolean hasValueChanged() {
            if (this.valueChangeNoticed) {
                this.valueChangeNoticed = false;
                return true;
            }
            return false;
        }

        protected int getMemoryUsed() {
            int result = 0;
            if (this.series != null) {
                result += this.series.getMemoryUsed();
            }
            return result;
        }

        protected void dump(PrintWriter stream) {
            this.calcStats();
            stream.print("  " + this.descriptor.getName() + "=");
            NumberFormat nf = StatArchiveReader.getNumberFormat();
            stream.print("[size=" + this.getSnapshotsSize() + " min=" + nf.format(this.min) + " max=" + nf.format(this.max) + " avg=" + nf.format(this.avg) + " stddev=" + nf.format(this.stddev) + "]");
            if (Boolean.getBoolean("StatArchiveReader.dumpall")) {
                this.series.dump(stream);
            } else {
                stream.println();
            }
        }

        protected void shrink() {
            this.series.shrink();
        }

        protected void initialValue(long v) {
            this.series.initialBits(v);
        }

        protected void prepareNextBits(long bits) {
            this.useNextBits = true;
            this.nextBits = bits;
        }

        protected void addSample() {
            this.statsValid = false;
            if (this.useNextBits) {
                this.useNextBits = false;
                this.series.addBits(this.nextBits);
                this.valueChangeNoticed = true;
            } else {
                this.series.addBits(0L);
            }
        }
    }

    protected static abstract class AbstractValue
    implements StatValue {
        protected StatDescriptor descriptor;
        protected int filter;
        protected long startTime = -1L;
        protected long endTime = -1L;
        protected boolean statsValid = false;
        protected int size;
        protected double min;
        protected double max;
        protected double avg;
        protected double stddev;
        protected double mostRecent;

        protected AbstractValue() {
        }

        public void calcStats() {
            if (!this.statsValid) {
                this.getSnapshots();
            }
        }

        @Override
        public int getSnapshotsSize() {
            this.calcStats();
            return this.size;
        }

        @Override
        public double getSnapshotsMinimum() {
            this.calcStats();
            return this.min;
        }

        @Override
        public double getSnapshotsMaximum() {
            this.calcStats();
            return this.max;
        }

        @Override
        public double getSnapshotsAverage() {
            this.calcStats();
            return this.avg;
        }

        @Override
        public double getSnapshotsStandardDeviation() {
            this.calcStats();
            return this.stddev;
        }

        @Override
        public double getSnapshotsMostRecent() {
            this.calcStats();
            return this.mostRecent;
        }

        @Override
        public StatDescriptor getDescriptor() {
            return this.descriptor;
        }

        @Override
        public int getFilter() {
            return this.filter;
        }

        @Override
        public void setFilter(int filter) {
            if (filter != this.filter) {
                if (filter != 0 && filter != 1 && filter != 2) {
                    throw new IllegalArgumentException(String.format("Filter value %s must be %s, %s, or %s.", filter, 0, 1, 2));
                }
                this.filter = filter;
                this.statsValid = false;
            }
        }

        protected void calcStats(double[] values) {
            if (this.statsValid) {
                return;
            }
            this.size = values.length;
            if (this.size == 0) {
                this.min = 0.0;
                this.max = 0.0;
                this.avg = 0.0;
                this.stddev = 0.0;
                this.mostRecent = 0.0;
            } else {
                int i;
                this.min = values[0];
                this.max = values[0];
                this.mostRecent = values[values.length - 1];
                double total = values[0];
                for (i = 1; i < this.size; ++i) {
                    total += values[i];
                    if (values[i] < this.min) {
                        this.min = values[i];
                        continue;
                    }
                    if (!(values[i] > this.max)) continue;
                    this.max = values[i];
                }
                this.avg = total / (double)this.size;
                this.stddev = 0.0;
                if (this.size > 1) {
                    for (i = 0; i < this.size; ++i) {
                        double dv = values[i] - this.avg;
                        this.stddev += dv * dv;
                    }
                    this.stddev /= (double)(this.size - 1);
                    this.stddev = Math.sqrt(this.stddev);
                }
            }
            this.statsValid = true;
        }

        public String toString() {
            this.calcStats();
            StringBuilder result = new StringBuilder();
            result.append(this.getDescriptor().getName());
            String units = this.getDescriptor().getUnits();
            if (units != null && units.length() > 0) {
                result.append(' ').append(units);
            }
            if (this.filter == 1) {
                result.append("/sec");
            } else if (this.filter == 2) {
                result.append("/sample");
            }
            result.append(": samples=").append(this.getSnapshotsSize());
            if (this.startTime != -1L) {
                result.append(" startTime=\"").append(new Date(this.startTime)).append("\"");
            }
            if (this.endTime != -1L) {
                result.append(" endTime=\"").append(new Date(this.endTime)).append("\"");
            }
            NumberFormat nf = StatArchiveReader.getNumberFormat();
            result.append(" min=").append(nf.format(this.min));
            result.append(" max=").append(nf.format(this.max));
            result.append(" average=").append(nf.format(this.avg));
            result.append(" stddev=").append(nf.format(this.stddev));
            result.append(" last=").append(nf.format(this.mostRecent));
            return result.toString();
        }
    }
}

