/*
 *
 * Purpose: Parses the libname from the table name, with the libname being the part before the
 *			first '.', and the table name being the part following.
 *
 * Parms:
 *		data= The libname.tablename designation
 *		part= Either TABLE or LIBRARY.  This is the part that is returned.  Defaults to TABLE.
 */ 
%macro SplitLibTable(data=, part=TABLE);

	%let outlib=WORK;

	%let splittablepos=%kindex(%nrbquote(&data.),.);
	%if ( &splittablepos. eq 0 ) %then 	%let outtable=%nrbquote(&data.);
	%else %do;
		%let outlib		=%ksubstr(%nrbquote(&data.),1,%eval(&splittablepos.-1));
		%let outtable  	=%nrbquote(%ksubstr(%nrbquote(&data.),%eval(&splittablepos.+1),%eval(%klength(%nrbquote(&data.))-&splittablepos.)));
	%end;

	%if (%lowcase("&PART.") eq "table") %then %do;
	  %nrbquote(&outtable.)
	%end;
	%else %do;
	  &outlib.
	%end;
%mend;

%macro DeleteDataSetIfExists(data=);

	%let libdel=%SplitLibTable(data=%nrbquote(&data.),part=LIBRARY);
	%let tbldel=%nrbquote(%SplitLibTable(data=%nrbquote(&data.),part=TABLE));
	%let nlittbl=%NLITERAL(&tbldel.);
	%let nliteral=&libdel..&nlittbl.;

	%if (%SYSFUNC(exist(&nliteral.))) %then %do;
		proc datasets library=&libdel. nolist;
			delete &nlittbl.;
		quit;
	%end;

%mend;

/* 
 * IsYes
 *
 * Purpose: Returns 1 if the option is set to YES, 0 otherwise
 *
 * Params: option Option Macro Variable
 *
 */
%macro IsYes( option );

   %IF ("%UPCASE(&&&option..)" eq "YES") %THEN %DO;
     1
   %END;
   %ELSE %DO;
     0
   %END;
%mend IsYes; 
/* 
 * NotYes
 *
 * Purpose: Returns 1 if the option is not set to YES, 1 otherwise
 *
 * Params: option Option Macro Variable
 *
 */
%macro NotYes( option );

   %IF ("%UPCASE(&&&option..)" ne "YES") %THEN %DO;
     1
   %END;
   %ELSE %DO;
     0
   %END;
%mend NotYes; 

/* 
 * SetOption
 *
 * Sets SAS option value on or off, remembering prior value for restore to original.
 *
 * Parameters
 * 	state	The state to which to set the option.  ON, OFF, or RESTORE.  Defaults to ON.
 * 			Off turns it off, on, turns it on, and restore restores the option to the 
 *			prior state before the last time it was set.
 *  option The option to set.
 *
 */
%macro SetOption( state, option);

	%IF ("&STATE." eq "") %then %return;
	%IF ("&OPTION." eq "") %then %return;

	%LET RESTOREVAR=RESTORE_&OPTION.;
	%global &RESTOREVAR.;

	%LET state=%upcase(&STATE.);
	%IF ( "&STATE." eq "RESTORE" ) %THEN %DO;
		options &&&RESTOREVAR..;
		%return;
	%END;

	%LET &RESTOREVAR.=%SYSFUNC(getoption(&OPTION.));
	%IF ( "&STATE." eq "ON" ) %THEN %DO;
		options &OPTION.
	%END;
	%ELSE %IF ( "&STATE." eq "OFF" ) %THEN %DO;
		options no&OPTION.
	%END;

%mend SetOption;

/*
 * DataSetNumObs
 * 
 * Retrieves the number of observations in a data set.
 * 
 * Params: memname The library.membername of the dataset to retrieve.
 */
%macro DataSetNumObs(memname);

    %if %str(%nrbquote(&memname.)) eq %str() %then
        %do;
            -1
            %* Cannot count observations when no member is specified;
            %return;
        %end;

    /* First, let's retrieve the number of observations, in case there are 0 observations in the dataset */
    %let dsname=%nrbquote(&memname.);
    %if %sysfunc(exist(%nrbquote(&dsname.))) %then
        %do;
        %let dsid=%sysfunc(open(%nrbquote(&dsname),i));
        %let numobs=%sysfunc(attrn(&dsid,nlobs));
        %let temp=%sysfunc(close(&dsid));
        %end;
    %else
        %let numobs=0;

    &numobs.

%mend DataSetNumObs;

/*
 * NLiteral
 * 
 * If needed, converts string provided to a SAS Name Literal.  Returns the result.
 * To work for names with spaces, this function expects the SAS system options
 * VALIDVARNAME=ANY and VALIDMEMNAME=EXTEND to be set.
 * 
 * Params: name The name to convert
 */
%macro NLiteral(memname);
	%KTRIM(%sysfunc(nliteral(&memname.)))
%mend;

/* 
 * SetOSSlash 
 * 
 * Sets the appropriate slash for the OS on which we are running into the macro variable OSSLASH.
 *
 */
%macro SetOSSlash;
	%GLOBAL OSSLASH;
	%IF (( &SYSSCP. eq WIN ) 
	  or ( &SYSSCP. eq DNTHOST )) %THEN %LET OSSLASH=%str(\);
	%ELSE %LET OSSLASH=%str(/);
%mend;

/*
 * CreateProcessIDFile
 *
 * Stores the job ID to an external file
 *
 * Params:
 *	pidfl=	The pid path and file name to create
 */
%macro CreateProcessIDFile(pidfl);

	%global AL_PIDFILE;
    data _null_;
	   file "&pidfl." PERMISSION='A::u::rw-,A::g::rw-,A::o::rw-';
	   put "&SYSJOBID.";
	run;

	%if ( &SYSERR. gt 4 ) %then %DumpError;
	%else
		/* Store away the PIDFILE for future reference */
		%let AL_PIDFILE=&pidfl.;

%mend;

/*
 * DeleteProcessIDFile
 * 
 * Removes the Process ID File if it exists.
 *
 */
%macro DeleteProcessIDFile;

   %if %SYMEXIST(AL_PIDFILE) %then %do;
	   	data _null_;
			fname="pidfile";
			rc=filename(fname, "&AL_PIDFILE.");
			if rc=0 and fexist(fname) then
				rc=fdelete(fname);
			rc=filename(fname);
		run;
		%if ( &SYSERR. eq 0 ) %then %SYMDEL AL_PIDFILE;
   %end;

%mend;

/*
 * GetCommandLine;
 * 
 * Retrieves the command-line used to launch this SAS session.
 *
 * Returns: The command-line string in the global macro variable CommandLine.
 */
%macro getcommandline; 
	%global CommandLine; 
	%if &sysscp=DNTHOST or &sysscp=WIN %then %do; 
		filename sascbtbl temp; 
		data _null_; file sascbtbl; 
		     put 'routine GetCommandLineA minarg=0 maxarg=0 returns=char'; 
		     put '        stackpop=called module=kernel32;'; 
		     run;
		data _null_; 
		     length command $32767;
		     command = modulec('GetCommandLineA'); 
		     call symput('CommandLine',ktrim(command)); 
		     run;
		filename sascbtbl clear; 
	%end; 
	%else %do; 
		data _null_; infile "/proc/&sysjobid./cmdline"; 
		     input; 
		     command=translate(_infile_,' ','00'x); 
		     call symput('CommandLine',ktrim(command)); 
		     stop; 
		     run;
	%end; 
%mend; 

/*
 * StripUnneededSASArgs
 * 
 * Cycles through a SAS command-line with arguments, removing arguments that are not required.
 *
 * Removes the following arguments:
 * -dms, -dmsexp
 * -log, -altlog
 * -print, -altprint
 * 
 * Params:
 * 	COMMAND=	The original command-line
 * 	DEBUG=		If set to 1, then debug output will be generated as the result.
 *				Defaults to 0.
 * 
 * Returns: The new (modified) command-line.  
 * 
 * Usage example:
 * 	%let NEWCOMMANDLINE=%StripUnneededSASArgs(COMMAND=&MYCOMMANDLINE.);
 */
%macro StripUnneededSASArgs(COMMAND=, DEBUG=0);

	%IF (&DEBUG.) %THEN %put ORIGINAL COMMAND: &COMMAND.;

	%let NEWCMD=;
	%let INQUOTE=0;
	%let SKIP=0;
	%let THISARG=;

	%let pos=1;
	%let delims=%str( );
	%let FRAG=%QSCAN(%nrbquote(&COMMAND.),&pos.,&delims.);
   	%DO %WHILE ("%nrbquote(&FRAG.)" ne ""); 

		%IF (&DEBUG.) %THEN %PUT %nrbquote(FRAGMENT #&pos.: &FRAG.);

		/* Are we opening a quote? */
		%if (
				%kindex(%str(&FRAG.),%str(%")) EQ 1 or
				%kindex(%str(&FRAG.),%str(%')) EQ 1
			) %then %let inquote=1;

		%if (%str(&INQUOTE.)) %then %do;
			%let THISARG=%str(&THISARG. &FRAG.);
		%end;
		%else
			%let THISARG=%str(&FRAG.);

		/* Are we closing a quote */
			%let LENGTH=%klength(%nrbquote(&FRAG.));
			%let LASTC=%qsubstr(&FRAG.,&LENGTH.,1);
			%LET DBLQUOTE=%str(%");
			%LET SNGQUOTE=%str(%');
		%if &DBLQUOTE. eq %quote(&LASTC.) or &SNGQUOTE. eq %quote(&LASTC.)
		%then %let inquote=0;

		/* Should we skip this one? */
		%if ( %str(&INQUOTE.) eq 0 ) %then %do;

			%IF (&DEBUG.) %THEN %put INQUOTE;
			%let UPARG=%kupcase(&THISARG.);
			%IF (&DEBUG.) %THEN %put &UPARG.;
			%if (
					%quote(&UPARG.) eq %str(-DMS) or
					%quote(&UPARG.) eq %str(-DMSEXP)
				) %then %let skip=1;
			%else %if (
					%kindex(%str(&UPARG.),%str(-LOG)) gt 0 or
					%kindex(%str(&UPARG.),%str(-ALTLOG)) gt 0 or
					%kindex(%str(&UPARG.),%str(-PRINT)) gt 0 or
					%kindex(%str(&UPARG.),%str(-ALTPRINT)) gt 0 
				) %then %let skip=2;
		%end;

		/* Skip unwanted arguments */
		%if (&inquote. eq 0) %then %do;
			%if (&skip. gt 0) %then %do;
				%let PRESKIP=&skip.;
				%let skip=%eval(&skip.-1);
				%IF (&DEBUG.) %THEN %put Changing SKIP &PRESKIP. -> &SKIP.;
			%end;
			%else %do;
				%let NEWCMD=%nrbquote(&NEWCMD. &THISARG.);
				%let THISARG=;
			%end;
		%end;

		%let POS=%eval(&pos.+1);
		%let FRAG=%QSCAN(%nrbquote(&COMMAND.),&pos.,&delims.);
	%END;
	%let NEWCMD=%KTRIM(&NEWCMD.);
	%IF (&DEBUG.) %THEN %PUT NEW COMMAND LINE=%nrbquote(&NEWCMD.);

	&NEWCMD.

%mend;

%macro normalizeSASNamesInDataSet( DATA=, COLUMN=filename, OUTCOLUMN=_normalname );

	%let DATA=%SUPERQ(DATA);

	data &DATA.;
		length &OUTCOLUMN. $256;

		set filelist;

		len=klength(&COLUMN.);
		put filename= length=;

		if (len>0) then do;
			/* Remove invalid characters */
			&OUTCOLUMN.=kcompress(filename,'/\*?<>|:-"');
			firstchar=ksubstr(&OUTCOLUMN.,1,1);
			/* Remove starting periods */
			do while ( (klength(&OUTCOLUMN.) > 0)  and
				 	  ((firstchar eq '.'      ) or
                       (firstchar eq ' '      )) );
				&OUTCOLUMN.=ksubstr(&OUTCOLUMN.,2);
				firstchar=ksubstr(&OUTCOLUMN.,1,1);
			end;
			/* Shorten to 32 characters */
			&OUTCOLUMN.=ksubstr(&OUTCOLUMN.,1,32);
			put &OUTCOLUMN.= firstchar=;
		end;
	run;

	/* Now that we have proposed normalized names, let's ensure they're unique */
	proc sort data=&DATA.;
	   by &OUTCOLUMN.;
	run;

	data &DATA.;
	   set &DATA.;
	   length _lastval $ 256;
	   retain _lastval ''  
              n 2;

	   /* Iterate filenames if there are name conflicts */
	   if (_lastval eq &OUTCOLUMN.) then do;
	   	  len=klength(&OUTCOLUMN.);
		  num="(" || put(n,3.) || ")";
		  num=" " || kcompress(num,' ');
		  numlen=klength(num);
		  if ( (len+numlen) > 32 ) then
	      	&OUTCOLUMN.=ksubstr(&OUTCOLUMN.,1,len-numlen) || num ;
		  else
		  	&OUTCOLUMN.=ktrim(&OUTCOLUMN.) || num;
		  n=n+1;
	   end;
	   else do;
	      n=2;
		  _lastval=&OUTCOLUMN.;
	   end;
	run;

%mend;

/*
 * Sleep
 * 
 * Sleeps for the number of seconds specified.
 *
 * Params:
 *    SECS	The number of seconds to sleep.  Defaults to 1.
 */
%macro sleep(secs);
	%setoption(OFF,NOTES);
	%if ("&secs." eq "") %then %let secs=1;
	data _null_;
		x=sleep(&secs.,1);
	run;
	%setoption(RESTORE,NOTES);
%mend;

/*
 * GetDateTimeString
 *
 * Retrieves the current date and time as a string into the variable specified
 *
 * Params:
 *    var	The macro variable into which to place the result
 *
 */
%macro GetDateTimeString( var=DATETIME );
	%global &var.;

	%setoption(off, notes);
	data _null_;
		dtstring=put(today(),nldate.) || " " || put(time(),TIMEAMPM.);
		call symput ("&var.",ktrim(dtstring));
	run;
	%setoption(restore, notes);


%mend;

/*
 * NormalizeStringForFilename
 *
 * Purpose: Normalizes the content of a string such that it contains only characters suitable
 *			for a file name.  Also replaces blanks with underscores.
 *			
 * Parms:
 *	_VAR=		The macro variable name containing the string to normalize.
 *
 * Returns:
 * 			Modifies the original macro variable by placingthe normalized value in the variable. 
 *
 * Usage example:
 * 
 *   %macro testmacro;
 *	
 *      %LET THESTR=%NRSTR(Joe%'s ..Library ><%%%"|&%(%));
 *      %put &THESTR.;
 * 
 *      %NormalizeStringForFilename(THESTR);
 *      %put &THESTR.;
 * 
 *   %mend;
 *   %testmacro;
 * 
 */
%macro NormalizeStringForFilename( _VAR );

	%let STR=%SUPERQ(&_VAR.);

	/* First, create initial normalized name proposal */
	%let _oldnotes=%sysfunc(getoption(NOTES));
	options nonotes;
	data _null_;

	    newstr=symget("&_VAR.");
		_norm_len=klength(newstr);

		if (_norm_len>0) then do;
			/* Remove invalid characters and compress result */
			outstr=ktranslate(newstr,'______________', ' /\&*?%<>|:-".');
			outstr=ktranslate(outstr,'_', "'");
			outstr=kcompress(outstr,'_');
			_norm_firstchar=ksubstr(outstr,1,1);
			/* Remove starting periods */
			do while ( (klength(outstr) > 0)  and
				 	  ((_norm_firstchar eq '.'      ) or
                       (_norm_firstchar eq ' '      )) );
				outstr=ksubstr(outstr,2);
				_norm_firstchar=ksubstr(outstr,1,1);
			end;
		end;
		call symput("&_VAR.", outstr);
	run;
	options &_oldnotes;

%mend;
/*
 * MetadataAssocLoopOpen
 * 
 * Purpose: Generates code to start a loop over the items within an association.
 *
 * Parms:
 * 		counter=	The counter variable to use while looping. Defaults to n.
 *		rc=			The return code variable to use when retrieving each item. Defaults to rc.
 *		starturi=	The uri variable containing the starting element for which the association exists. 
 *					Defaults to starturi.  
 *		assocnam=	The name of the association for which elements should be retrieved.
 *					This value must be in single quotes.  For example, 'SASLibrary'.
 *		itemuri=	The variable which will store the uri of each item.  Defaults to itemuri.
 *		type=		The variable which will store the type of each item.  Defaults to type.
 *		id=			The variable which will store the id of each item.  Defaults to id.
 * 
 *	Notes:
 *		This macro is intended for use inside of a data step.
 *		Variables used by this macro must have their lengths defined before calling the macro.
 */
%macro MetadataAssocLoopOpen( counter=n, rc=rc, starturi=starturi, assocname=, itemuri=, type=type, id= );

	&counter.=1;
	call missing( &itemuri. );
	&rc.=metadata_getnasn(&starturi.,&assocname.,&counter., &itemuri.);
	do while (&rc.>0);
		call missing(&type.,&id.);
		rc=metadata_resolve(&itemuri.,&type.,&id.);

%mend;

/*
 * MetadataAssocLoopClose
 * 
 * Purpose: Generates code to close a loop opened by %MetadataAssocLoopOpen.
 *
 * Parms:
 * 		counter=	The counter variable to use while looping. Defaults to n.
 *		rc=			The return code variable to use when retrieving each item. Defaults to rc.
 *		starturi=	The uri variable containing the starting element for which the association exists. 
 *					Defaults to starturi.  
 *		assocnam=	The name of the association for which elements should be retrieved.
 *					This value must be in single quotes.  For example, 'SASLibrary'.
 *		itemuri=	The variable which will store the uri of each item.  Defaults to itemuri.
 * 
 *	Notes:
 *		This macro is intended for use inside of a data step.
 *		Variables used by this macro must have their lengths defined before calling the macro.
 */
%macro MetadataAssocLoopClose( counter=n, rc=rc, starturi=starturi, assocname=, itemuri= );

		&counter.=&counter.+1;
		&rc.=metadata_getnasn(&starturi.,&assocname.,&counter.,&itemuri.);
	end; 

%mend;

/*
 * RenameColumnByIndex
 * 
 * Purpose: Changes the name of a particular column number (by index) to a new name
 *
 * Parms:
 * 		data=		The data set containing the column to be renamed.
 *		colindex=	The column index of the column to be renamed.
 *		newcolname=	The new column name.
 */
%macro RenameColumnByIndex( DATA=, COLINDEX=5, NEWCOLNAME=LastModified );

	/* Get name of column to rename by index */
	%IF %SYMEXIST(COLTORENAME) %THEN %SYMDEL COLTORENAME;

	%LET TABLENAME=%SplitLibTable(data=%nrbquote(&DATA.),PART=TABLE);
	%LET LIBNAME=%SplitLibTable(data=%nrbquote(&DATA.),PART=LIBRARY);

	%LET TMPCOLRENAME=WORK.TMPCOLRENAME;
	proc sql noprint;
		create table &TMPCOLRENAME. as
		 	select name 
		 	from dictionary.columns
		 	where memname=upcase("&TABLENAME.");
	quit;
	data _null_;
		set &TMPCOLRENAME.;
		if ( _N_ = &COLINDEX. ) and ( name ne "&NEWCOLNAME." ) then
		   call symput("COLTORENAME", ktrim(name) );
	run;

	%if %SYMEXIST(COLTORENAME) %then %do;
		proc datasets library=&LIBNAME. nolist;
			modify &TABLENAME.;
			rename "&COLTORENAME."n=&NEWCOLNAME.;
		quit;
	%end;
%mend;
