/*
 * MetadataExtensionValue
 * 
 * Purpose: Retrieves/Sets the value of a metadata extension property for an object.
 * 
 * Parms:
 * 	Type=	The metadata object type of the object for which the extension property value is desired
 * 	Name=	The metadata name of the object for which the extension property value is desired
 *	Key=	The key of the extension for the value is desired
 *  Value=	If set, the metadata key is SET TO the value specified. 
 *			Otherwise, the key value is returned in retvar.
 *  Retvar=	The name of the macro variable into which to return the key value.
 *			If Value is specified, then the value of the key is SET.
 *
 * Returns:
 *  If VALUE= specified, returns MISSING always.
 *  If VALUE= unspecified, the macro variable specified in RETVAR= will contain the value of
 *	the KEY= extension. MISSING if not found.
 *
 */
%macro MetadataExtensionValue( Type=, Name=, Key=, RetVar=MEVRC, value= );

	%if ("&TYPE." eq "") %then %do;
		%put ERROR: TYPE= is required to retrieve extension value;
		%return;
	%end;

	%if ("&NAME." eq "") %then %do;
		%put ERROR: NAME= is required to retrieve extension value;
		%return;
	%end;

	%if ("&KEY." eq "") %then %do;
		%put ERROR: KEY= is required to retrieve extension value;
		%return;
	%end;

	%global &RETVAR.;
	%let &RETVAR.=MISSING;
	%let query = %str(omsobj:&type.?@Name=%quote(%')&Name%quote(%'));

	%setoption(off,notes);
	data _null_;

		length 
			objuri 
			assocuri
			value
			tmp
			key
			$256

			associd
			$17

			type
			$32
			;

		call missing(objuri,assocuri,tmp,type);
		key="&KEY.";

		nobj=1;
		objrc=metadata_getnobj("&query.",nobj,objuri);
		if (objrc <1) then do;
			put "No matching &TYPE. objects found. (RC=" objrc ")";
			stop;
		end;

		/* Get value */

		/* We have found the object.   Now let's find the extension value */
		n=1;
		assocrc=metadata_getnasn(objuri,'Extensions',n, assocuri);
		do while (assocrc>0);
			call missing(value,associd);
			rc=metadata_resolve(assocuri,type,associd);

			if (type='Extension') then do;
				rc=metadata_getattr(assocuri,'Name',tmp);
				if (tmp = key) then do;
					rc=metadata_getattr(assocuri,'Value',value);
					call symput("&RETVAR.",ktrim(value));
					put KEY "=" VALUE;
				end;
			end; 

			n=n+1;
			assocrc=metadata_getnasn(objuri,'Extensions',n,assocuri);
		end; /* do while (assocrc>0) */

	run;
	%setoption(restore,notes);

%mend;

/*
 * GetSoftwareComponentKeyValue
 * 
 * Purpose: Retrieves the key value of a sofware component key/value pair.
 * 
 * Parms:
 * 	ClassID=The class identifier of the software component desired
 *	Key=	The key for which the value is desired
 *  Retvar=	The name of the macro variable into which to return the key value.
 *
 * Returns:
 *  The macro variable specified in RETVAR= will contain the value of
 *	the KEY= specified. MISSING if not found.
 *
 */
%macro GetSoftwareComponentKeyValue( ClassID=, Key=, RetVar=MEVRC );

	%if ("&CLASSID." eq "") %then %do;
		%put ERROR: CLASSID= is required to retrieve software component value;
		%return;
	%end;

	%if ("&KEY." eq "") %then %do;
		%put ERROR: KEY= is required to retrieve software component value;
		%return;
	%end;

	%global &RETVAR.;
	%let &RETVAR.=MISSING;
	%let query = %str(omsobj:SoftwareComponent?@ClassIdentifier=%quote(%')&CLASSID.%quote(%'));

	data _null_;

		length 
			objuri 
			assocuri
			assuri
			value
			tmp
			key
			$256

			assid
			associd
			$17

			type
			$32
			;

		call missing(objuri,assocuri,assid,assuri,tmp,type,p,name);
		key="&KEY.";

		nobj=1;
		objrc=metadata_getnobj("&query.",nobj,objuri);
		if (objrc <1) then do;
			put "No matching SoftwareComponent objects found. (RC=" objrc ")";
			stop;
		end;

		/* We have found the object.   Now let's find the key value */
		n=1;
		assocrc=metadata_getnasn(objuri,'PropertySets',n, assocuri);
		do while (assocrc>0);
			call missing(value,associd);
			rc=metadata_resolve(assocuri,type,associd);

			if (type='PropertySet') then do;

				rc=metadata_getattr(assocuri,'Name',tmp);

				if (tmp='Application Configuration') then do;

			        /* Walk through all software component application config properties */
					nass=1;
					rc=1;
				    do while(rc>0);
						rc=metadata_getnasn(assocuri,'SetProperties',n, assuri);
						c=metadata_resolve(assuri,type,assid);
						c=metadata_getattr(assid,'Name',tmp);

						if (tmp="&KEY.") then 
						do;
							c=metadata_getattr(assid,'DefaultValue',value);
							call symput("&RETVAR.",ktrim(value));
							put "&KEY.=" VALUE;

						end;
				        n=n+1;
				    end;
				end;
			end; 

			n=n+1;
			assocrc=metadata_getnasn(objuri,'PropertySets',n,assocuri);
		end; /* do while (assocrc>0) */
	run;

%mend;
