/*
 * LASRLibraryMetadataToDataSet
 * 
 * Purpose: Takes a LASR library name, finds it in metadata, and loads associated metadata to a data set.
 *
 * Parms:
 * 		data= The output data set to create.  If unspecified, defaults to work.lasrlib.
 *		name= The name of the LASR library to retrieve.  This name must be unique in the metadata server.
 *
 */
%macro LASRLibraryMetadataToDataSet(data=WORK.LASRLIB, name=);

	%if ("&name." eq "" ) %then %do;
	   %msg( type=ERROR,msg=%str(Library name must be specified with NAME=));
	   %return;
	%end;

	%let query = %str(omsobj:SASLibrary?@Engine contains 'SASIOLA' and @Name=%quote(%')&name.%quote(%'));

	data &data.;

	/* The KEEP statement defines the variables to include in the output data set.  */
	  keep libid libname libengine libref libtag smp servname lasrhost lasrport lasrinst path signer lasrmonpath
		   tablemem externalmem lifetime force nodes
		   loggingenabled loggingpath loggingmaxfilesize loggingmaxrollover loggingkeeplog loggingadditional
           serververbose dplibid dplibname dplibengine dplibref dppath ros_enabled ros_tbldefault ros_method metafolder tablefullcopies;

	/* The LENGTH statement defines variables for function arguments and
	assigns the maximum length of each variable.  */

	  length 
	   		libid 
			dplibid
			ttransid 
			tsrcid 
			pathid $17

			/* Library properties */
			libname
			dplibname $128 
			libengine
			dplibengine
			libref 
			dplibref $8
			libtag $32

			/* Server properties */
			smp $8
			lasrhost $256 
			lasrport $8 
			signer 
			lasrmonpath $256
			lasrinst 
			path 
			dppath $256
			force $5
			nodes $8

			tablemem 
			externalmem $3
			lifetime			$32

			loggingenabled $8
			loggingpath $256
			loggingmaxfilesize
			loggingmaxrollover  
			loggingkeeplog		$8
			loggingadditional	$128

			serververbose	$8

			ros_enabled $3
			ros_tbldefault $3
			ros_method $9

			tablefullcopies $5

			/* Library extension properties */
			extid $17
			exttype $32
			extname $128
			extvalue $128

			metafolder $256

			liburi 
			connuri

			ttransnuri
			tsrcnuri
			pathnuri
			extnuri

			upasnuri 
			propnuri
			servnuri 
			servname $256

			type 
			ttranstype 
			tsrctype 
			pathtype $32

			tmp $256
			;

	/* The CALL MISSING routine initializes the output variables to missing values.  */
	  call missing(lasrinst,path,liburi,servnuri,servname,propnuri,connuri,upasnuri,libname,libengine,libref,lasrhost,lasrport,libtag,tmp,signer,lasrmonpath,smp,force,nodes);
	  call missing(dplibid,dplibname,dplibengine,dplibref,dppath);
	  call missing(tablemem,externalmem,lifetime,loggingenabled,loggingpath,loggingmaxfilesize,loggingmaxrollover,loggingkeeplog,loggingadditional,serververbose);
	  call missing(ros_enabled,ros_tbldefault,ros_method,metafolder,tablefullcopies);
	  call missing(extname,extvalue);

	  /* Query for the Library Object */
	  nlibobj=1;
	  librc=metadata_getnobj("&query.",nlibobj,liburi);
	  if (librc <1) then put librc " matching libraries found.";

	  /* Default certain values in case they are not present in metadata */
	  if ( librc > 0 ) then do;
	     libtag='WORK';
	  end;

	  /* Process the library object */
	  do while (librc>0);

	     /* Get Library attributes */
	     rc=metadata_getattr(liburi,'Name',libname);
	     rc=metadata_getattr(liburi,'Engine',libengine);
		 rc=metadata_getattr(liburi,'Libref',libref);
		 rc=metadata_getattr(liburi,'Id',libid);
		 put libengine "library " libname;
		 put @7 "LIBREF=" libref;	 

		 /* Get Library properties */
		 n=1;
		 proprc=metadata_getnasn(liburi,'Properties',n, propnuri);
		 do while (proprc>0);
	    if proprc > 0 then do;
	       call missing(type,libid);
	       rc=metadata_resolve(propnuri,type,libid);

	       if type='Property' then do;
		   	  rc=metadata_getattr(propnuri,'Name',tmp);
			  if (tmp='Library.LASR.Property.Tag.Name.xmlKey.txt') then do;
		      	rc=metadata_getattr(propnuri,'DefaultValue',libtag);
			    put @7 "TAG=" libtag;
			  end;
			  else
			  	put "NOTE: Unprocessed Library property " tmp;
		   end; 

	       n+1;
	       proprc=metadata_getnasn(liburi,'Properties',n,propnuri);
		 end; /* if proprc > 0 */
		end; /* do while (proprc>0) */

		/* Get Library Extension Values */
		%MetadataAssocLoopOpen( counter=extn, rc=extrc, starturi=liburi, 
								 assocname='Extensions', itemuri=extnuri, 
								 type=exttype, id=extid )

		if (exttype='Extension') then do;
			rc=metadata_getattr(extnuri,'Name',extname);
			rc=metadata_getattr(extnuri,'Value',extvalue);
			if (kupcase(extname) = kupcase("&EXT_KEY_ROS_ENABLED.")) then
				ros_enabled = extvalue;
			else if (kupcase(extname) = kupcase("&EXT_KEY_ROS_TBLDEFAULT.")) then
				ros_tbldefault = extvalue;
			else if (kupcase(extname) = kupcase("&EXT_KEY_ROS_METHOD.")) then
				ros_method = extvalue;
			else if (kupcase(extname) = kupcase("&EXT_KEY_METAFOLDER.")) then
				metafolder = extvalue;
			else if (kupcase(extname) = kupcase("&EXT_KEY_TABLE_FULLCOPIES.")) then
			    tablefullcopies = extvalue;
				
		end; 

		%MetadataAssocLoopClose( counter=extn, rc=extrc, starturi=liburi, 
								 assocname='Extensions', itemuri=extnuri )

		/* Get Library's LASR Server Connection */
	    uprc=metadata_getnasn(liburi,'LibraryConnection',1,upasnuri);
		if uprc > 0 then do;

		/* When a LibraryConnection association is found, the METADATA_RESOLVE function 
		is called to resolve the URI to an object on the metadata server. */

	    call missing(type,libid,lasrhost,lasrport);
	    rc=metadata_resolve(upasnuri,type,libid);
		    
		/* Get Connection Attributes */
		if type='SASClientConnection' then do;

	      put type;
	      rc=metadata_getattr(upasnuri,'Port',lasrport);
	      put @7 "Port " lasrport;

			/* Get Connection Properties */
			n=1;
			proprc=metadata_getnasn(upasnuri,'Properties',n, propnuri);
			do while (proprc>0);
				if proprc > 0 then do;
					call missing(type,libid);
					rc=metadata_resolve(propnuri,type,libid);

					if type='Property' then do;
						rc=metadata_getattr(propnuri,'PropertyName',tmp);
						if (tmp='SIGNER') then do;
							rc=metadata_getattr(propnuri,'DefaultValue',signer);
							put @7 "SIGNER=" signer;
						end;
						else if (tmp='HOST') then do;
							rc=metadata_getattr(propnuri,'DefaultValue',lasrhost);
							put @7 "HOST=" lasrhost;
						end;
						else
							put "Unprocessed Connection property " tmp;
					end; 

					n+1;
					proprc=metadata_getnasn(upasnuri,'Properties',n,propnuri);
					end; /* if proprc > 0 */
				end; /* Property processing */

				/* Get Library's LASR Server */
				n=1;
			    servrc=metadata_getnasn(upasnuri,'Source',n,servnuri);
				if servrc > 0 then do;
				    call missing(type,smp);
				    rc=metadata_resolve(servnuri,type,libid);
					    
					/* Get Server Attributes */
					if type='ServerComponent' then do;

				      rc=metadata_getattr(servnuri,'Name',servname);
				      put type servname;

					  /* Get Server Properties */

						n=1;
						proprc=metadata_getnasn(servnuri,'Properties',n, propnuri);
						do while (proprc>0);
							if proprc > 0 then do;
								call missing(type,libid);
								rc=metadata_resolve(propnuri,type,libid);

								if type='Property' then do;
									rc=metadata_getattr(propnuri,'PropertyName',tmp);
									if (tmp='INSTALL') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',lasrinst);
										put @7 "INSTALL=" lasrinst;
									end;
									else if (tmp='PATH') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',path);
										put @7 "PATH=" path;
									end;
									else if (tmp='FORCE') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',force);
										put @7 "FORCE=" force;
									end;
									else if (tmp='NODES') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',nodes);
										put @7 "NODES=" nodes;
									end;
									else if (tmp='SMPMODE') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',smp);
										put @7 "SMP=" smp;
									end;
									else if (tmp='TABLEMEM') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',tablemem);
										put @7 tmp "=" tablemem;
									end;
									else if (tmp='EXTERNALMEM') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',externalmem);
										put @7 tmp "=" externalmem;
									end;
									else if (tmp='LIFETIME') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',lifetime);
										put @7 tmp "=" lifetime;
									end;
									else if (tmp='VERBOSE') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',serververbose);
										put @7 tmp "=" serververbose;
									end;
									else if (tmp='LOGGING') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',loggingenabled);
										put @7 tmp "=" loggingenabled;
									end;
									else if (tmp='LOGPATH') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',loggingpath);
										put @7 tmp "=" loggingpath;
									end;
									else if (tmp='MAXFILESIZE') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',loggingmaxfilesize);
										put @7 tmp "=" loggingmaxfilesize;
									end;
									else if (tmp='MAXROLLNUM') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',loggingmaxrollover);
										put @7 tmp "=" loggingmaxrollover;
									end;
									else if (tmp='KEEPLOG') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',loggingkeeplog);
										put @7 tmp "=" loggingkeeplog;
									end;
									else if (tmp='ADDTIONALLOGOPTIONS') then do;
										rc=metadata_getattr(propnuri,'DefaultValue',loggingadditional);
										put @7 tmp "=" loggingadditional;
									end;
									else
										put "Unprocessed Server property " tmp;
								end; /* type='Property' */

								n+1;
								proprc=metadata_getnasn(servnuri,'Properties',n,propnuri);
							end; /* if proprc > 0 */
						end; /* Property processing */

						/* Get Server Extensions Values */
						%MetadataAssocLoopOpen( counter=extn, rc=extrc, starturi=servnuri, 
												 assocname='Extensions', itemuri=extnuri, 
												 type=exttype, id=extid )

						if (exttype='Extension') then do;
							rc=metadata_getattr(extnuri,'Name',extname);
							rc=metadata_getattr(extnuri,'Value',extvalue);
							if (kupcase(extname) = kupcase("&EXT_KEY_MON_PATH.")) then
								lasrmonpath = extvalue;
						end; 

						%MetadataAssocLoopClose( counter=extn, rc=extrc, starturi=servnuri, 
												 assocname='Extensions', itemuri=extnuri )

					end; /* type='Server Component' */
				end; /* servrc>0 */

			end; /* Connection processing */ 

		end; /* if uprc > 0 */

		/* Get Data Provider Library */
		dpfound=0;
		%MetadataAssocLoopOpen( counter=ttransn, rc=ttransrc, starturi=liburi, 
								 assocname='TargetTransformations', itemuri=ttransnuri, 
								 type=ttranstype, id=ttransid )

			if ( ttranstype='Transformation' ) then do;

				rc=metadata_getattr(ttransnuri,'TransformRole',tmp);
				if (tmp='DataProviderLibrary') then do;

					%MetadataAssocLoopOpen( counter=tsrcn, rc=tsrcrc, starturi=ttransnuri, 
											 assocname='TransformationSources', itemuri=tsrcnuri, 
											 type=tsrctype, id=tsrcid )

					rc=metadata_getattr(tsrcnuri,'Engine',tmp );
					if ( tmp='BASE' ) then do;

						dpfound=1;
						dplibengine=tmp;
						rc=metadata_getattr(tsrcnuri,'Id',dplibid );
						rc=metadata_getattr(tsrcnuri,'Name',dplibname );
						rc=metadata_getattr(tsrcnuri,'Libref',dplibref );

						%MetadataAssocLoopOpen( counter=pathn, rc=pathrc, starturi=tsrcnuri, 
												 assocname='UsingPackages', itemuri=pathnuri, 
												 type=pathtype, id=pathid )
						/* Get path; */
						if ( pathtype = "Directory" ) then 
							rc=metadata_getattr(pathnuri,'DirectoryName',dppath );

						%MetadataAssocLoopClose( counter=pathn, rc=pathrc, starturi=tsrcnuri, 
												 assocname='UsingPackages', itemuri=pathnuri )

						put dplibengine "library " dplibname;
						put @7 "LIBREF=" dplibref;	 
						put @7 "PATH=" dppath;	 

						/* Found data provider library.  Leave loop */
						leave;
					end; /* BASE Engine */

					%MetadataAssocLoopClose( counter=tsrcn, rc=tsrcrc, starturi=ttransnuri, 
											 assocname='TransformationSources', itemuri=tsrcnuri )

						leave;
				end; /* DataProviderLibrary */

			end; /* ttranstype='Transformation' */

		%MetadataAssocLoopClose( counter=ttransn, rc=ttransrc, starturi=liburi, 
								 assocname='TargetTransformations', itemuri=ttransnuri )
		if ( dpfound=0 ) then put 'No data provider library associated with LASR library.';

		  output;
		 /* Look for another library */
		 nlibobj+1;
	     librc=metadata_getnobj("&query.",nlibobj,liburi);
	  end; /* do while (librc>0) */
	run;

%mend;
