/*
 * GetMetadataTablesList
 * 
 * Purpose: Takes a library name, finds it in metadata, outputs information about the tables associated
 *          to it to a SAS data set.  This macro works for database macros.
 *
 * Parms:
 * 		data= The output data set to create.  If unspecified, defaults to work.tbllist.
 *		name= The name of the library to query.  This name must be unique in the metadata server.
 *		libref= The libref to use when building the fullref output field
 */
%macro GetMetadataTablesList( data=WORK.TBLLIST, name=, libref=WORK );

	%if ("&name." eq "" ) %then %do;
	   %put ERROR: Library name must be specified with NAME=;
	   %return;
	%end;

	%let query = %str(omsobj:SASLibrary?@Name='&name.');

	data &data.;

	/* The KEEP statement defines the variables to include in the output data set.  */
	  keep fullref libid libname libengine libref libtag tblid tblname sastblname lasrcompress ros_enabled;

	/* The LENGTH statement defines variables for function arguments and
	assigns the maximum length of each variable.  */

	  length 

	  		fullref $137
				
	   		libid
			pkgid
					$17

			/* Library properties */
			libname $128 
			libengine
			libref $8
			libtag $32

			/* Table properties */
			tblid $17
			propsetsid $17
			extnsid $17
			tblname $128
			sastblname $32
			lasrcompress $3
			ros_enabled $3

			/* Properties properties */
			propsetname $128
			propertyname $128
			propertyvalue $128

			/* Extensions properties */
			extname $128
			extvalue $128

			uplibname $256

			liburi 
			upasnuri 
			tablesnuri 
			propsetsnuri
			extnsnuri
						$256

			type $32
			;

	/* The CALL MISSING routine initializes the output variables to missing values.  */
	  call missing(libid,liburi,upasnuri,libname,libengine,libref,libtag,uplibname);

	  /* Query for the Library Object */
	  nlibobj=1;
	  librc=metadata_getnobj("&query.",nlibobj,liburi);
	  if (librc <1) then put librc " matching libraries found.";

	  /* Process the library object */
	  do while (librc>0);

	     /* Get Library attributes */
	     rc=metadata_getattr(liburi,'Name',libname);
	     rc=metadata_getattr(liburi,'Engine',libengine);
	     rc=metadata_getattr(liburi,'libref',libref);
		 rc=metadata_getattr(liburi,'Id',libid);
*		 put "Found " libengine "library " libname "with ID " libid;

		/* Get Library's UsingPackages association */
	    uprc=metadata_getnasn(liburi,'UsingPackages',1,upasnuri);
		if uprc > 0 then do;

			/* When a UsingPackages association is found, the METADATA_RESOLVE function 
			is called to resolve the URI to an object on the metadata server. */

		    call missing(type,pkgid,tablesnuri);
		    rc=metadata_resolve(upasnuri,type,pkgid);
			    
			/* Get DatabaseSchema Attributes */
			if type='DatabaseSchema' then do;

		      rc=metadata_getattr(upasnuri,'Name',uplibname);
*		      put @7 "Found " type "Name " uplibname;

			  /* If we found the database schema to match the libname,
			     let's go for the tables. */
			  if ( uplibname eq libname ) then do;
				nTablesObj=1;
*				put "Retrieving metadata tables in library " libname "...";
				tablesrc=metadata_getnasn(upasnuri,'Tables',nTablesObj, tablesnuri);
				do while (tablesrc>0);
					call missing(type,tblid,tblname,sastblname,propsetsnuri,extnsnuri,ros_enabled);
					lasrcompress="NO";

				    rc=metadata_resolve(tablesnuri,type,tblid);

					rc=metadata_getattr(tablesnuri,'Name',tblname);
					rc=metadata_getattr(tablesnuri,'SASTableName',sastblname);
					fullref="&libref.."||sastblname;

					*put "Table #" nTablesObj " (" tblid "):" sastblname "(" tblname ")";

					/* Retrieve properties from the table */
					nPropSetsObj=1;
					propsetsrc=metadata_getnasn(tablesnuri,'PropertySets',nPropSetsObj, propsetsnuri);
					do while (propsetsrc>0);

						call missing(propsetsid, propsetname);
		    			rc=metadata_resolve(propsetsnuri,type,propsetsid);

						rc=metadata_getattr(propsetsnuri, 'Name', propsetname );

						if propsetname='LASR Properties' then do;

							nPropertyObj=1;
							*put "Retrieving properties from property set " propsetname "...";

							call missing(propertyname,type,propertyvalue);
							propertyrc=metadata_getnprp(propsetsnuri,nPropertyObj, propertyname, propertyvalue);
							do while (propertyrc>0);
								if (kupcase(propertyname)="COMPRESS") then 
									lasrcompress=kupcase(propertyvalue);
								nPropertyObj=nPropertyObj+1;
								propertyrc=metadata_getnprp(propsetsnuri,nPropertyObj, propertyname, propertyvalue);
							end;
						end;

						nPropSetsObj=nPropSetsObj+1;
						propsetsrc=metadata_getnasn(tablesnuri,'PropertySets',nPropSetsObj, propsetsnuri);
					end;

					/* Retrieve extension settings from the table */
					nExtnsObj=1;
					extnsrc=metadata_getnasn(tablesnuri,'Extensions',nExtnsObj, extnsnuri);
					do while (extnsrc>0);
						call missing(extname,extvalue,extnsid);
						rc=metadata_resolve(extnsnuri,type,extnsid);

						if (type='Extension') then do;
							rc=metadata_getattr(extnsnuri,'Name',extname);
							rc=metadata_getattr(extnsnuri,'Value',extvalue);
							if (kupcase(extname) = 'VA.RELOADONSTART.ENABLED') then do;
								rc=metadata_getattr(extnsnuri,'Value',extvalue);
								if ( kupcase(extvalue)= "YES" or extvalue="1" ) then
								   ros_enabled = "YES";
*								put extname "=" extvalue;
							end;
						end; 

						nExtnsObj=nExtnsObj+1;
						extnsrc=metadata_getnasn(tablesnuri,'Extensions',nExtnsObj, extnsnuri);
					end;

					output;

					nTablesObj=nTablesObj+1;
					tablesrc=metadata_getnasn(upasnuri,'Tables',nTablesObj, tablesnuri);
				end; /* tablesrc > 0 */

			  end; /* Found correct schema uplibname eq libname */
			end; /* DatabaseSchema processing */ 

		end; /* if uprc > 0 */

		 /* Look for another library */
		 nlibobj+1;
	     librc=metadata_getnobj("&query.",nlibobj,liburi);
	  end; /* do while (librc>0) */
	run;

	proc sort data=&data.;
	   by sastblname tblname;
	run;

%mend;
