/*
 * GetLASRServerLibrariesList
 * 
 * Purpose: Takes a LASR server, finds it in metadata, and outputs limited information 
 *          about the libraries associated with it to a SAS data set.  
 *
 * Parms:
 * 		data= The output data set to create.  If unspecified, defaults to work.liblist.
 *		name= The name of the LASR server to query.  This name must be unique in the 
 *            metadata server.
 *		libref= The libref to use when building the fullref output field
 */
%macro GetLASRServerLibrariesList( data=WORK.TBLLIST, name= );

	%if ("&name." eq "" ) %then %do;
	   %put ERROR: LASR server name must be specified with NAME=;
	   %return;
	%end;

	%let query = %str(omsobj:ServerComponent?@Name='&name.');

	data &data.;

	/* The KEEP statement defines the variables to include in the output data set.  */
	keep servid servname libid libname ros_enabled dp_assigned;

	/* The LENGTH statement defines variables for function arguments and
	assigns the maximum length of each variable.  */

	  length 

			/* server properties */
	   		servid $17
			servname $128 

			/* connection properties */
	   		connid $17
			connname $128 
			conntype $32

			/* library properties */
	   		libid $17
			libname $128 
			libtype $32

			/* extension properties */
			extid $17
			exttype $32
			extname $128
			extvalue $128

			/* transformation properties */
			ttransid $17
			ttranstype $32

			tsrcid $17
			tsrctype $32

			ros_enabled $3
			dp_assigned $3

			/* uri's */
			servuri 
			connnuri
			libnuri
			extnuri
			ttransnuri
			tsrcnuri

			tmp
						$256

			;

	  /* The CALL MISSING routine initializes the output variables to missing values.  */
	  call missing(servid,servuri,servname,libid,libname,extid,connname,ros_enabled,extname,extvalue,tmp);

	  /* Query for the Server Object */
	  nservobj=1;
	  servrc=metadata_getnobj("&query.",nservobj,servuri);
	  if (servrc <1) then put servrc " matching servers found.";

	  /* Process the server object */
	  do while (servrc>0);

	     /* Get server attributes */
	     rc=metadata_getattr(servuri,'Name',servname);
		 rc=metadata_getattr(servuri,'Id',servid);
		 *put "Server:";
		 *put @3 servname @70 servid;

		/* Get connections */
		%MetadataAssocLoopOpen( counter=connn, rc=connrc, starturi=servuri, 
								 assocname='SourceConnections', itemuri=connnuri, 
								 type=conntype, id=connid )
		/* Get connection */
		if ( conntype = 'SASClientConnection' ) then do;
			rc=metadata_getattr(connnuri,'Id',connid);
			rc=metadata_getattr(connnuri,'Name',connname );
			*put @3 connname @70 connid;

			/* Get libraries */
			%MetadataAssocLoopOpen( counter=libn, rc=librc, starturi=connnuri, 
									 assocname='Libraries', itemuri=libnuri, 
									 type=libtype, id=libid )

			/* Get library*/
			if ( libtype = 'SASLibrary' ) then do;
				rc=metadata_getattr(libnuri,'Id',libid);
				rc=metadata_getattr(libnuri,'Name',libname );
				*put @6 libname @70 libid;

				call missing(ros_enabled);

				/* Get extensions */
				%MetadataAssocLoopOpen( counter=extn, rc=extrc, starturi=libnuri, 
										 assocname='Extensions', itemuri=extnuri, 
										 type=exttype, id=extid )

				if (exttype='Extension') then do;
					rc=metadata_getattr(extnuri,'Name',extname);
					rc=metadata_getattr(extnuri,'Value',extvalue);
					if (kupcase(extname) = 'VA.RELOADONSTART.ENABLED') then do;
						rc=metadata_getattr(extnuri,'Value',extvalue);
						ros_enabled = extvalue;
						*put @9 extname "=" extvalue;
					end;
				end; 

				%MetadataAssocLoopClose( counter=extn, rc=extrc, starturi=libnuri, 
										 assocname='Extensions', itemuri=extnuri )

				/* Determine if data provider library is assigned */
				dp_assigned='No';
				%MetadataAssocLoopOpen( counter=ttransn, rc=ttransrc, starturi=libnuri, 
										 assocname='TargetTransformations', itemuri=ttransnuri, 
										 type=ttranstype, id=ttransid )

					if ( ttranstype='Transformation' ) then do;

						rc=metadata_getattr(ttransnuri,'TransformRole',tmp);
						if (tmp='DataProviderLibrary') then do;

							%MetadataAssocLoopOpen( counter=tsrcn, rc=tsrcrc, starturi=ttransnuri, 
													 assocname='TransformationSources', itemuri=tsrcnuri, 
													 type=tsrctype, id=tsrcid )

							rc=metadata_getattr(tsrcnuri,'Engine',tmp );
							if ( tmp='BASE' ) then do;
								dp_assigned='Yes';
								*put @9 'Data provider library assigned';
								/* Found data provider library.  Leave loop */
								leave;
							end; /* BASE Engine */

							%MetadataAssocLoopClose( counter=tsrcn, rc=tsrcrc, starturi=ttransnuri, 
													 assocname='TransformationSources', itemuri=tsrcnuri )
							leave;
						end; /* DataProviderLibrary */
					end; /* ttranstype='Transformation' */

				%MetadataAssocLoopClose( counter=ttransn, rc=ttransrc, starturi=libnuri, 
										 assocname='TargetTransformations', itemuri=ttransnuri )
				/*if ( dp_assigned ne 'Yes' ) then put @9 'No data provider library associated with LASR library.';*/

				output;
			end;

			%MetadataAssocLoopClose( counter=libn, rc=librc, starturi=connnuri, 
									 assocname='Libraries', itemuri=libnuri )
		end;

		%MetadataAssocLoopClose( counter=connn, rc=connrc, starturi=servuri, 
								 assocname='SourceConnections', itemuri=connnuri )

		 /* Look for another server */
		 nservobj+1;
	     servrc=metadata_getnobj("&query.",nservobj,servuri);
	  end; /* do while (servrc>0) */
	run;

	proc sort data=&data.;
	   by servname libname;
	run;

%mend;
