#!/bin/sh

# ------------------------------------------------------------------------------
# Copyright (c) 2021 - 2023 by SAS Institute Inc., Cary, NC USA  27513
# ------------------------------------------------------------------------------
#
# Invoke Java functionality to encrypt and base64 encode a value.
#
# WARNING: A known bug in Apache Commons CLI (CLI-275) prevents values with
# embedded blanks from making it through the parser on Linux hosts, even though
# the value appears on the command line as a quoted string.  The first word of
# the quoted string is taken as the argument value and the rest of the string is
# seemingly ignored.  Curiously, quoted strings work as expected on Windows.
#
# This means...
#
# On Linux, a value to be encrypted and encoded cannot have embedded blanks.
#
# Environment variables JAVA_HOME and CATALINA_HOME must already be defined.
#
# ------------------------------------------------------------------------------

ERROR_STATE=0
DEBUG_MODE=0


function realpath
{
   #
   # Determine the absolute path equivalent for a specified path
   #
   REAL_PATH=
   if [ -d "$1" ]; then
      CDIR=`/bin/pwd`
      \cd "$1"
      REAL_PATH=`pwd -P`
      \cd $CDIR 
   else
      echo "ERROR: The real path not exist for directory: $1"
   fi
}


for arg in "$@"; do
    shift
    if [ "$arg" = "-debug" ]; then
        DEBUG_MODE=1
        echo "DEBUG mode is enabled for the script only."
    else
        set -- "$@" "$arg"
    fi
done


if [ $DEBUG_MODE -ne 0 ]; then
    echo "DEBUG: Number of command line arguments = $#"
    COUNT=1
    for arg in "$@"; do
        echo "$COUNT = $arg"
        COUNT=`expr $COUNT + 1`
    done
fi

if [ -z "$JAVA_HOME" ]; then
    echo "ERROR: The JAVA_HOME environment variable is not defined"
    ERROR_STATE=1
else
    if [ $DEBUG_MODE -ne 0 ]; then
        echo "DEBUG: JAVA_HOME = $JAVA_HOME"
    fi
fi

if [ -z "$CATALINA_HOME" ]; then
    echo "ERROR: The CATALINA_HOME environment variable is not defined"
    ERROR_STATE=1
else
    if [ $DEBUG_MODE -ne 0 ]; then
        echo "DEBUG: CATALINA_HOME = $CATALINA_HOME"
    fi
fi


if [ $ERROR_STATE -eq 0 ]; then

    realpath `dirname "$0"`
    SRC_DIR=$REAL_PATH    
    if [ $DEBUG_MODE -ne 0 ]; then
        echo "DEBUG: SRC_DIR = $SRC_DIR"
    fi

    realpath "$SRC_DIR/../lib"
    LIB_DIR=$REAL_PATH

    if [ ! -d $LIB_DIR ]; then
        echo "ERROR: The library directory does not exist: $LIB_DIR"
        ERROR_STATE=1
    else
        HOME_LIB=$CATALINA_HOME/lib
        if [ ! -d $HOME_LIB ]; then
            echo "ERROR: The library directory does not exist: $HOME_LIB"
            ERROR_STATE=1
        else
            if [ $DEBUG_MODE -ne 0 ]; then
                echo "DEBUG: LIB_DIR  = $LIB_DIR"
                echo "DEBUG: HOME_LIB = $HOME_LIB"
            fi

            CLASSPATH=$SRC_DIR:$LIB_DIR:$HOME_LIB
            for JAR_FILE in $LIB_DIR/*.jar; do
                CLASSPATH=$CLASSPATH:$JAR_FILE
            done

            realpath "$SRC_DIR/../log4j2"
            LOG4J_DIR=$REAL_PATH
            CLASSPATH=$CLASSPATH:$LOG4J_DIR/conf/log4j2.xml
            for JAR_FILE in $LOG4J_DIR/lib/*.jar; do
                CLASSPATH=$CLASSPATH:$JAR_FILE
            done

            for JAR_FILE in $HOME_LIB/*.jar; do
                CLASSPATH=$CLASSPATH:$JAR_FILE
            done

            if [ $DEBUG_MODE -ne 0 ]; then
                echo "DEBUG: CLASSPATH = $CLASSPATH"
            fi

            if [ $# -eq 0 ]; then
                HELP=-help
            fi

            "$JAVA_HOME/bin/java" -classpath $CLASSPATH com.sas.vfabrictcsvr.decoder.RunDecoder $HELP $@
            ERROR_STATE=$?

            if [ $ERROR_STATE -ne 0 ]; then
                echo "Run with no arguments or the --help switch to see usage information"
            fi
        fi
    fi
fi

exit $ERROR_STATE
