# ------------------------------------------------------------------------------
# Copyright (c) 2021 - 2023 by SAS Institute Inc., Cary, NC USA 27513
# ------------------------------------------------------------------------------
#
# PUROPSE:  Create/install, uninstall, start, or stop a Windows service that
#           executes a specified Tomcat web application server.  A list of
#           services whose names match a specified pattern can also displayed.
#
# NOTES:
#
# ------------------------------------------------------------------------------


<#
.SYNOPSIS
Manage SAS Web Application servers as a Windows service.

.DESCRIPTION
The SAS Web Application servers can be executed as a Windows service, allowing
them to run without anyone having to login to the machine.  This utility is
used to install and uninstall a server as a service, and can also be used to
start and stop the execution of the service and the server it is hosting.

.PARAMETER Action
REQUIRED - The operation to perform on a Windows Service hosting a server.
Possible operations are: list, install, uninstall, start, and stop.

    LIST - Display a table of properties for the specified services.
    INSTALL - Install a new Windows service.
    UNINSTALL - Uninstall a Windows service (WARNING: cannot be undone!).
    START - Start execution of an installed service.
    STOP - Stop execution of an installed service.

    NOTE: The INSTALL action requires that environment variables CATALINA_HOME
    and CATALINA_BASE both be defined, and that the paths they specify exist.
    In addition, a Java home directory must be specified either by the JAVA_HOME
    environment variable or the JavaHome command line option.

.PARAMETER Name
REQUIRED - The short name of the Windows service to operate on.  A short name
that makes it easy to refer to the service.

WARNING: Using a service name that contains blanks or non-alphanumeric characters
is likely to lead to difficulties in the future when specifying the service name
to command-line utilities like this one. 

.PARAMETER Display
OPTIONAL - The service name to be displayed (such as in the Windows Services
control panel).  This is typically a long name that more fully reflects the
attributes of the service.  Defaults to "SAS WebAppServer - <Name>".

.PARAMETER Description
OPTIONAL - A description of the service's purpose, benefits, attributes, etc.
Defaults to "SAS WebAppServer - Apache Tomcat web application server".

.PARAMETER JavaHome
OPTIONAL - The full path to the root of the Java installation directory.
This parameter is not necessary when the JAVA_HOME environment variable is
defined, but is required if it is not.

.PARAMETER NoExec
OPTIONAL - Gather all inputs and make all preparations for normal execution, but
do not actually execute the operation.  The final command that would have been
executed is displayed as output.

.PARAMETER Debug
OPTIONAL - Output additional messages showing run-time parameters and values.

.PARAMETER Verbose
OPTIONAL - Output additional messages showing more in-depth information than would
normally be shown.

.EXAMPLE  
#
#
# Output a table showing all services whose display name is 'like' the pattern
# in the specified name.  Wildcard characters are not needed or accepted in the
# name value.
#
# Use this command to see the name of the service (as opposed to the display-name
# as shown in the Windows Services panel) and the process ID if the service is
# executing.  If necessary, the process ID can be used to kill a process and the
# server that is executing within it by:   TASKKILL /F /PID pid_number
#
SASServiceCtl.ps1 -action list -name SASServer

Service Name                                  Service Display Name                           Run State Memory  Process ID
------------                                  --------------------                           --------- ------  ----------
SAS-Config-Lev4-SASServer1_1-WebAppServer     SAS [Config-Lev4] SASServer1_1 - WebAppServer  Running   3.02 GB       6088
SAS-Config-Lev4-SASServer12_1-WebAppServer    SAS [Config-Lev4] SASServer12_1 - WebAppServer Stopped   4 KB             0
SAS-Config-Lev4-SASServer2_1-WebAppServer     SAS [Config-Lev4] SASServer2_1 - WebAppServer  Stopped   4 KB             0


.EXAMPLE
#
#
# Service installation with all service properties specified on the command line
# using typical SAS installer values.
#
SASServiceCtl.ps1 -action install -name SAS-Config-Lev1-SASServer1_1-WebAppServer -display "SAS [Config-Lev1] SASServer1_1 - WebAppServer" -description "SASWebApp - Apache Tomcat web application server on port 8080"

NOTE: The 'SAS-Config-Lev1-SASServer1_1-WebAppServer' service has been installed.


.EXAMPLE
#
#
# Bare minimum service installation.  The service display name and its description
# are set to defaults, and the JAVA_HOME environment variable must be defined.
#
#   Default display name: SAS WebAppServer - <Name>
#   Default description:  SAS WebAppServer - Apache Tomcat web application server
#
SASServiceCtl.ps1 -action install -name SAS-Config-Lev1-SASServer1_1-WebAppServer

NOTE: The 'SAS-Config-Lev1-SASServer1_1-WebAppServer' service has been installed.


.EXAMPLE
#
#
# Service installation with a specific Java home directory specified.
#
SASServiceCtl.ps1 -action install -name SASServer1_1 -javahome "C:\Program Files\SASHome\SASPrivateJavaRuntimeEnvironment\9.4\jre"

NOTE: The 'SASServer1_1' service has been installed.


.EXAMPLE
#
#
# Invoking a PowerShell environment from the Windows command line and passing
# it the name of the SAS service control script to execute, and the script
# arguments, without actually executing the specified action.
#
powershell -file .\SASServiceCtl.ps1 -action install -name SAS-Config-Lev4-SASServer2_1-WebAppServer -debug -verbose -noexec

DEBUG: Debug mode is enabled
VERBOSE: Verbose mode is enabled
VERBOSE: NoExec mode is enabled
DEBUG: Execution directory = E:\SAS\Config\Lev4\Web\WebAppServer\SASServer2_1\bin
DEBUG: Action = UNINSTALL
DEBUG: Name = SAS-Config-Lev4-SASServer2_1-WebAppServer
DEBUG: CATALINA_HOME = C:\Program Files\SASHome\SASWebApplicationServer\9.4\apache-tomcat-9.0.55
DEBUG: Tomcat executable = C:\Program Files\SASHome\SASWebApplicationServer\9.4\apache-tomcat-9.0.55\bin\Tomcat9.exe
Simulated execution of: & "C:\Program Files\SASHome\SASWebApplicationServer\9.4\apache-tomcat-9.0.55\bin\Tomcat9.exe" "//DS//SAS-Config-Lev4-SASServer2_1-WebAppServer"


.EXAMPLE
#
#
# Uninstalling an existing service from the Windows command line by
# invoking PowerShell to execute the SAS service control script.
#
powershell -file SASServiceCtl.ps1 -action uninstall -name SAS-Config-Lev1-SASServer2_1-WebAppServer

NOTE: The 'SAS-Config-Lev1-SASServer2_1-WebAppServer' service has been uninstalled.


.EXAMPLE
#
#
# Start an existing service that is not running.
#
SASServiceCtl.ps1 -action start -name SAS-Config-Lev1-SASServer1_1

C:\SAS\Config\Lev1\Web\WebAppServer\SASServer1_1\bin> powershell -file SASServiceCtl.ps1 -action start -name SAS-Config-Lev1-SASServer1_1-WebAppServer
WARNING: Waiting for service 'SAS [Config-Lev1] SASServer1_1 - WebAppServer (SAS-Config-Lev1-SASServer1_1-WebAppServer)' to start...
WARNING: Waiting for service 'SAS [Config-Lev1] SASServer1_1 - WebAppServer (SAS-Config-Lev1-SASServer1_1-WebAppServer)' to start...
WARNING: Waiting for service 'SAS [Config-Lev1] SASServer1_1 - WebAppServer (SAS-Config-Lev1-SASServer1_1-WebAppServer)' to start...
NOTE: The 'SAS-Config-Lev1-SASServer1_1-WebAppServer' service has been started.


.EXAMPLE
#
#
# Stop a service that is running.
#
SASServiceCtl.ps1 -action stop -name SAS-Config-Lev1-SASServer1_1

C:\SAS\Config\Lev1\Web\WebAppServer\SASServer1_1\bin> powershell -file SASServiceCtl.ps1 -action stop -name SAS-Config-Lev4-SASServer1_1-WebAppServer
NOTE: The 'SAS-Config-Lev1-SASServer1_1-WebAppServer' service has been stopped.

#>



[ CmdletBinding ( PositionalBinding = $false ) ]

Param ( [Parameter ( Mandatory = $true,
                     HelpMessage = "(Required) Operation to perform: list | install | uninstall | start | stop"
                   )
        ]
        [validateset ( "List", "Install", "Uninstall", "Start", "Stop" ) ]
        [ValidateNotNullOrEmpty()]
        [String] $Action,
        # ----------------------------------------------------------------------
        [Parameter ( Mandatory = $true,
                     HelpMessage = "(Required) Name of the service to operate on"
                   )
        ]
        [ValidateNotNullOrEmpty()]
        [String] $Name,
        # ----------------------------------------------------------------------
        [Parameter ( Mandatory = $false,
                     HelpMessage = "(Optional) Long name to display for the service"
                   )
        ]
        [String] $Display,
        # ----------------------------------------------------------------------
        [Parameter ( Mandatory = $false,
                     HelpMessage = "(Optional) Full description of the service"
                   )
        ]
        [String] $Description,
        # ----------------------------------------------------------------------
        [Parameter ( Mandatory = $false,
                     HelpMessage = "(Optional) Full path to the root of the Java install directory"
                   )
        ]
        [String] $JavaHome,
        # ----------------------------------------------------------------------
        [Parameter ( Mandatory = $false,
                     HelpMessage = "(Optional) Prepare to execute, but do not do so"
                   )
        ]
        [Switch] $NoExec
      )


Set-Variable DEFAULT_DISPLAY_VALUE     -Option Constant -Value "SAS WebAppServer - $Name"
Set-Variable DEFAULT_DESCRIPTION_VALUE -Option Constant -Value "SAS WebAppServer - Apache Tomcat web application server"
Set-Variable TOMCAT_EXECUTABLE_NAME    -Option Constant -Value "Tomcat9.exe"


function ValidateEnvPath ( [String] $EnvVar )
{
    [String] $PathValue = [System.Environment]::GetEnvironmentVariable($EnvVar,'Process')
    if ( $PathValue.Length -eq 0 )
    {
        Write-Error "ERROR: The $EnvVar environment variable is not defined." `
        -Category ObjectNotFound
    }

    if ( (Test-Path -Path $PathValue -PathType Container) -eq $false )
    {
        Write-Error -Message "ERROR: The $EnvVar path was not found: $PathValue" `
                    -Category ObjectNotFound
    }
    return $PathValue
}


function GetEnvVars ( [String] $WrapperFileName )
{
    [String] $NonPathEnvVars = ""
    [String] $SystemPathAdditions = ""

    if ( (Test-Path -Path $WrapperFileName -PathType Leaf) -eq $false )
    {
        Write-Error -Message "ERROR: File not found: $WrapperFileName" `
                    -Category ObjectNotFound
    }

    else
    {
        foreach ( $Line in ( Get-Content $WrapperFileName ) )
        {
            # NOTE: The PATH= definitions are handled separately because they
            # are cummlative in nature where the other environment variable
            # definitions are not.  There may be multiple set.PATH= statements
            # in the file, and each adds to the list of paths.  For all other
            # environment variables, additional statements for the same variable
            # adds a new definition of the variable that will likely replace any
            # previous definition at run-time.

            if ( $Line -match "^\s*set\.path\s*=\s*.*" )
            {
                [String] $Path = ( $Line -split "^\s*set\.path\s*=\s*" )
                $Path = $Path.Trim()
                if ( $Path.Length -gt 0 )
                {
                    # Eliminate all quotes around a single path value.  The entire path
                    # string will be wrapped in single quotes at the end of the routine.
                    $Path = $Path -replace "`"", "" -replace "'", ""

                    $PathCount++
                    if ( $PathCount -eq 1 )
                    {
                        $PathParsed = "System paths parsed from file ${WrapperFileName}:`n"
                    }
                    $PathParsed += "   $PathCount $Path`n"

                    if ( $PathCount -ne 1 )
                    {
                        $SystemPathAdditions += ( ";" )
                    }
                    $SystemPathAdditions += $Path
                }
            }

            elseif ( $Line -match "^\s*set\.(.*)\s*=\s*.*" )
            {
                $Line = $Line -replace "`"", "" -replace "'", ""
                [String] $EnvName, $EnvValue = $Line -replace "set\.", "" -split '=', 2
                $EnvName  = $EnvName.Trim()
                $EnvValue = $EnvValue.Trim()
                if ( ( $EnvName.Length -gt 0 ) -and ( $EnvValue.Length -gt 0 ) )
                {
                    $EnvCount++
                    if ( $EnvCount -eq 1 )
                    {
                        $EnvParsed = "Non-path environment variables parsed from file ${WrapperFileName}:`n"
                    }

                    else
                    {
                        $NonPathEnvVars = "${NonPathEnvVars}#"
                    }
                    $EnvParsed += "   ${EnvCount} ${EnvName} = ${EnvValue}`n"
                    $NonPathEnvVars = "${NonPathEnvVars}${EnvName}=${EnvValue}"
                }
            }
        }

        if ( $SystemPathAdditions.Length -gt 0 )
        {
            # Wrap the entire string in single quotes.
            $SystemPathAdditions = "'$SystemPathAdditions'"
        }

        if ( $PathCount -eq 0 )
        {
            Write-Verbose "No system paths were parsed from $WrapperFileName"
        }

        elseif ( -not ( $PathParsed.IsNullOrEmpty ) )
        {
            Write-Verbose "$PathParsed`n"
        }

        if ( -not ( $EnvParsed.IsNullOrEmpty ) )
        {
            Write-Verbose "$EnvParsed`n"
        }
    }

    return $NonPathEnvVars, $SystemPathAdditions
}


function GetJVMOptions ( [String] $WrapperFileName )
{
    [String] $JVMOptionList = ""

    if ( (Test-Path -Path $WrapperFileName -PathType Leaf) -eq $false )
    {
        Write-Error -Message "ERROR: File not found: $WrapperFileName" `
                    -Category ObjectNotFound
    }

    else
    {
        $Count = 0
        $Parsed = ""

        foreach ( $Line in ( Get-Content $WrapperFileName ) )
        {
            if ( $Line -match "^\s*wrapper.java.additional.\d+\s*=\s*.*" )
            {
                [String] $Option = ( $Line -split "^\s*wrapper.java.additional.\d+\s*=\s*" )
                $Option = $Option.Trim()
                if ( $Option.Length -gt 0 )
                {
                    # Replace double-quotes around the entire option with
                    # single-quotes around just the option value.
                    if ( $Option -match "^`".*`"$" )
                    {
                        $Option = $Option -replace '"', ''
                        [String] $OptName, [String] $OptValue = $Option -split "="
                        $Option = "$OptName=`'$OptValue`'"
                    }

                    # Replace all double-quotes used in a single option with single-quotes.
                    $Option = $Option -replace "`"", "'"

                    # Wrap all semi-colons used in a single option in single-quotes.
                    $Option = $Option -replace ";", "';'"

                    $Count++
                    if ( $Count -eq 1 )
                    {
                        $Parsed = "JVM options parsed from file ${WrapperFileName}:`n"
                    }
                    $Parsed += "   $Count $Option`n"

                    if ( $Count -ne 1 )
                    {
                        # Add an option separator (semi-colon) to the end of the previous
                        # option value, prior to adding the new option.  Adding the separator
                        # prior to appending a new option, rather than adding the separator as
                        # part of the previous option prevents a troublesome trailing separator
                        # appearing at the end of the option string.
                        $JVMOptionList += ( ";" )
                    }
                    $JVMOptionList += $Option
                }
            }
        }

        if ( $Count -eq 0 )
        {
            Write-Warning "No JVM options were parsed from $WrapperFileName"
        }

        elseif ( -not ( $Parsed.IsNullOrEmpty ) )
        {
            Write-Verbose "$Parsed`n"
        }
    }

    return $JVMOptionList
}


function GetLibraryPath ( [String] $WrapperFileName )
{
    [String] $LibPath = ""

    if ( (Test-Path -Path $WrapperFileName -PathType Leaf) -eq $false )
    {
        Write-Error -Message "ERROR: File not found: $WrapperFileName" `
                    -Category ObjectNotFound
    }

    else
    {
        $Count = 0
        $Parsed = ""

        foreach ( $Line in ( Get-Content $WrapperFileName ) )
        {
            if ( $Line -match "^\s*wrapper.java.library.path.\d+\s*=\s*.*" )
            {
                [String] $Path = ( $Line -split "^\s*wrapper.java.library.path.\d+\s*=\s*" )
                $Path = $Path.Trim()
                if ( $Path.Length -gt 0 )
                {
                    # Replace all double-quotes around a single path with single-quotes.
                    $Path = $Path -replace "`"", "'"

                    $Count++
                    if ( $Count -eq 1 )
                    {
                        $Parsed = "Library paths parsed from file ${WrapperFileName}:`n"
                    }
                    $Parsed += "   $Count $Path`n"

                    if ( $Count -ne 1 )
                    {
                        $LibPath += ( ";" )
                    }
                    $LibPath += $Path
                }

            }
        }

        if ( $Count -eq 0 )
        {
            Write-Warning "No library paths were parsed from $WrapperFileName"
        }

        elseif ( -not ( $Parsed.IsNullOrEmpty ) )
        {
            Write-Verbose "$Parsed`n"
        }        
    }

    return $LibPath
}


function GetClassPath ( [String] $WrapperFileName )
{
    [String] $ClassPath = ""

    if ( (Test-Path -Path $WrapperFileName -PathType Leaf) -eq $false )
    {
        Write-Error -Message "ERROR: File not found: $WrapperFileName" `
                    -Category ObjectNotFound
    }

    else
    {
        $Count = 0
        $Parsed = ""

        foreach ( $Line in ( Get-Content $WrapperFileName ) )
        {
            if ( $Line -match "^\s*wrapper.java.classpath.\d+\s*=\s*.*" )
            {
                [String] $Path = ( $Line -split "^\s*wrapper.java.classpath.\d+\s*=\s*" )
                $Path = $Path.Trim()
                if ( $Path.Length -gt 0 )
                {
                    # Replace all double-quotes around a single path with single-quotes.
                    $Path = $Path -replace "`"", "'"

                    $Count++
                    if ( $Count -eq 1 )
                    {
                        $Parsed = "Class paths parsed from file ${WrapperFileName}:`n"
                    }
                    $Parsed += "   $Count $Path`n"

                    if ( $Count -ne 1 )
                    {
                        $ClassPath += ( ";" )
                    }
                    $ClassPath += $Path
                }
            }
        }

        if ( $Count -eq 0 )
        {
            Write-Warning "No class paths were parsed from $WrapperFileName"
        }

        elseif ( -not ( $Parsed.IsNullOrEmpty ) )
        {
            Write-Verbose "$Parsed`n"
        }
    }

    return $ClassPath
}


function Format-Bytes ( [Float] $Value )
{
   $MemSizes = 'KB','MB','GB','TB','PB'

   for ( $i = 0; $i -lt $MemSizes.count; $i++ )
   {
      if ( $Value -lt "1$($MemSizes[$i])" )
      {
         if ( $i -eq 0 )
         {
            return "$Value B"
         }

         else
         {
            $Value = $Value / "1$($MemSizes[$i-1])"
            $Value = "{0:n2}" -f $Value
            return "$Value $($MemSizes[$i-1])"
         }
      }
   }
}


function Main()
{
    $InitialDebugPreference = $DebugPreference
    $InitialVerbosePreference = $VerbosePreference
    $InitialErrorActionPreference = $ErrorActionPreference

    $ErrorActionPreference = "Stop"

    $NoExecMode = $false
    [String] $JavaHomeDir = ""
    [String] $TomcatExe = ""
    $Status = 0


    #
    # Report message and execution options if they are used.
    #
    if ( $PSCmdlet.MyInvocation.BoundParameters['Debug'].IsPresent )
    {
        $DebugPreference = 'Continue'
        Write-Debug 'Debug mode is enabled'
    }

    if ( $PSCmdlet.MyInvocation.BoundParameters['Verbose'].IsPresent )
    {
        $VerbosePreference = 'Continue'
        Write-Verbose 'Verbose mode is enabled'
    }

    if ( $NoExec.IsPresent )
    {
        Write-Verbose 'NoExec mode is enabled'
        $NoExecMode = $true
    }

    #
    # Standardize the action command to upper case.
    #
    $Action = $Action.ToUpper()


    Write-Debug "Execution directory = $PSScriptRoot"
    Write-Debug "Action = $Action"
    Write-Debug "Name = $Name"


    #
    # Warn if the users account does not have Administrator privileges.
    #
    if ( -NOT ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()`
              ).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator") )
    {
        Write-Warning "You are not running this script under an account that has Administrator privileges.`
               It is possible that some actions will fail when not running as an Administraor."
    }


    if ( ( $Action -eq "INSTALL" ) -or ( $ACTION -eq "UNINSTALL") )
    {
        #
        # Ensure CATALINA_HOME is defined and exists.
        #
        [String] $CatalinaHome = ValidateEnvPath ( "CATALINA_HOME" )
        Write-Debug "CATALINA_HOME = $CatalinaHome"

        #
        # Ensure the Tomcat utility for Windows Services exists.
        #
        $TomcatExe = "$CatalinaHome\bin\$TOMCAT_EXECUTABLE_NAME"
        if ( (Test-Path -Path $TomcatExe -PathType Leaf) -eq $false )
        {
            Write-Error -Message "ERROR: File not found: $TomcatExe" `
                        -Category ObjectNotFound
        }
        Write-Debug "Tomcat executable = $TomcatExe"
    }

    if ( $Action -eq "INSTALL" )
    {
        #
        # Ensure CATALINA_BASE is defined and exists.
        #
        [String] $CatalinaBase = ValidateEnvPath ( "CATALINA_BASE" )
        Write-Debug "CATALINA_BASE = $CatalinaBase"

        #
        # Ensure JAVA is defined and exists.
        #
        if ( -not $JavaHome )
        {
            [String] $JavaHomeEnv = $Env:JAVA_HOME
            if ( $JavaHomeEnv.Length -eq 0 )
            {
                Write-Error -Message "ERROR: The JAVA_HOME environment variable is not defined, and the `
              JavaHome option was not specified.  One or the other is required." `
                -Category ObjectNotFound
            }

            else
            {
                Write-Debug "JAVA_HOME = $JavaHomeEnv"
                $JavaHomeDir = $JavaHomeEnv
            }
        }

        else
        {
            Write-Debug "JavaHome = $JavaHome"
            $JavaHomeDir = $JavaHome
        }

        $FilePath = "$JavaHomeDir\bin\java.exe"
        if ( (Test-Path -Path $FilePath -PathType Leaf) -eq $false )
        {
            Write-Error -Message "ERROR: File not found: $FilePath" `
                        -Category ObjectNotFound
        }

        $FilePath = "$JavaHomeDir\bin\server\jvm.dll"
        if ( (Test-Path -Path $FilePath -PathType Leaf) -eq $false )
        {
            Write-Error -Message "ERROR: File not found: $FilePath" `
                        -Category ObjectNotFound
        }
    }


    #
    # Perform the specified ACTION.
    #
    switch ( $Action )
    {
        "LIST"
        {
            $Name = $Name -replace '\*', '' -replace '\?', '' -replace ';', ''

            $Command = "Get-WmiObject Win32_Service -Filter `"DisplayName like '%$Name%'`" |`
                        Sort-Object -Property DisplayName |`
                        Select-Object Name, DisplayName, State, ProcessID |`
                        Format-Table"

            if ( $NoExecMode -eq $true )
            {
                # NOTE: This is close, but not the whole command being executed.
                # The actual processing is more involved so that process memory
                # can be included in the output table.
                Write-Output "Simulated execution of: $Command"
            }

            else
            {
               $ServiceList = Get-WmiObject Win32_Service -Filter "DisplayName like '%$Name%'" | Sort-Object -Property DisplayName
               $ServicePID = $ServiceList.ProcessID
               $ServiceProcessList = foreach ($Process in $ServicePID) { Get-WmiObject Win32_Process -Filter "ProcessId = '$Process'" }
               $ProcessMemList = $ServiceProcessList.WS

               $Results = @()
               for ( $i = 0; $i -lt $ServiceList.count; $i++ )
               {
                   $Obj = New-Object PSObject
                   Add-Member -InputObject $Obj -MemberType NoteProperty -Name Name            -Value $ServiceList[$i].Name
                   Add-Member -InputObject $Obj -MemberType NoteProperty -Name DisplayName     -Value $ServiceList[$i].DisplayName
                   Add-Member -InputObject $Obj -MemberType NoteProperty -Name State           -Value $ServiceList[$i].State
                   Add-Member -InputObject $Obj -MemberType NoteProperty -Name ProcessId       -Value $ServiceList[$i].ProcessId
                   Add-Member -InputObject $Obj -MemberType NoteProperty -Name MemoryRaw       -Value $ProcessMemList[$i]
                   Add-Member -InputObject $Obj -MemberType NoteProperty -Name MemoryFormatted -Value ( "{0:n0}" -f $ProcessMemList[$i] )
                   Add-Member -InputObject $Obj -MemberType NoteProperty -Name MemoryScaled    -Value ( Format-Bytes ( [Float] $ProcessMemList[$i] ) )
                   $Results = $Results + $Obj
               }

               $Fmt = @{Expression={$_.Name};         Label="Service Name"}, `
                      @{Expression={$_.DisplayName};  Label="Service Display Name"}, `
                      @{Expression={$_.State};        Label="Run State"}, `
                      @{Expression={$_.MemoryScaled}; Label="Memory"}, `
                      @{Expression={$_.ProcessId};    Label="Process ID"}

               $Results | Select-Object Name, DisplayName, State, MemoryScaled, ProcessID| Format-Table $Fmt
            }
        }

        "INSTALL"
        {
            [String] $EnvironmentVars, $SystemPathAdditions = GetEnvVars ( "${CatalinaBase}\conf\wrapper.conf" )

            $EnvironmentVars = $EnvironmentVars -replace '%CATALINA_HOME%', $CatalinaHome
            $EnvironmentVars = $EnvironmentVars -replace '%CATALINA_BASE%', $CatalinaBase
            $EnvironmentVars = $EnvironmentVars -replace '%JAVA_HOME%', $JavaHome
            $EnvironmentVars = $EnvironmentVars -replace '%JRE_HOME%', $JavaHome

            $SystemPathAdditions = $SystemPathAdditions -replace '%CATALINA_HOME%', $CatalinaHome
            $SystemPathAdditions = $SystemPathAdditions -replace '%CATALINA_BASE%', $CatalinaBase
            $SystemPathAdditions = $SystemPathAdditions -replace '%JAVA_HOME%', $JavaHome
            $SystemPathAdditions = $SystemPathAdditions -replace '%JRE_HOME%', $JavaHome

            if ( $SystemPathAdditions )
            {
                Write-Verbose "System path additions string:`n$SystemPathAdditions`n`n"
                $EnvironmentVars = "$EnvironmentVars#PATH=$SystemPathAdditions"
            }

            Write-Verbose "Environment variables defined:`n$EnvironmentVars`n`n"

            [String] $JVMOptionList = GetJVMOptions ( "${CatalinaBase}\conf\wrapper.conf" )
            $JVMOptionList = $JVMOptionList -replace '%CATALINA_HOME%', $CatalinaHome
            $JVMOptionList = $JVMOptionList -replace '%CATALINA_BASE%', $CatalinaBase
            $JVMOptionList = $JVMOptionList -replace '%JAVA_HOME%', $JavaHome
            $JVMOptionList = $JVMOptionList -replace '%JRE_HOME%', $JavaHome

            [String] $JavaLibraryPath = GetLibraryPath ( "${CatalinaBase}\conf\wrapper.conf" )
            $JavaLibraryPath = $JavaLibraryPath -replace '%CATALINA_HOME%', $CatalinaHome
            $JavaLibraryPath = $JavaLibraryPath -replace '%CATALINA_BASE%', $CatalinaBase
            $JavaLibraryPath = $JavaLibraryPath -replace '%JAVA_HOME%', $JavaHome
            $JavaLibraryPath = $JavaLibraryPath -replace '%JRE_HOME%', $JavaHome   

            if ( $JavaLibraryPath )
            {
                Write-Verbose "JVM library path string:`n$JavaLibraryPath`n`n"                                                                

                if ( $JVMOptionList -match "-Djava.library.path=" )
                {
                   Write-Output "WARNING: JVM option java.library.path already defined.  Ignoring library paths from wrapper.conf."
                   $JavaLibraryPath = ""
                }
                
                else
                {
                    $JVMOptionList = $JVMOptionList + ";-Djava.library.path='" + $JavaLibraryPath + "'"
                }
            }

            if ( $JVMOptionList )
            {
                Write-Verbose "JVM option string:`n$JVMOptionList`n`n"
            }

            #
            # Provide defaults for certain JVM options if not already specified.
            #
            [String] $JvmMs_Option = "--JvmMs 1024 "
            if ( $JVMOptionList -match "-Xms\d+" )
            {
                $JvmMs_Option = ""
            }
            Write-Debug "JvmMs_Option = $JvmMs_Option`n"
            

            [String] $JmvMx_Option = "--JvmMx 4096 "
            if ( $JVMOptionList -match "-Xmx\d+" )
            {
                $JmvMx_Option = ""
            }
            Write-Debug "JmvMx_Option = $JmvMx_Option`n"


            [String] $ClassPath = GetClassPath ( "${CatalinaBase}\conf\wrapper.conf" )
            $ClassPath = $ClassPath -replace '%CATALINA_HOME%', $CatalinaHome
            $ClassPath = $ClassPath -replace '%CATALINA_BASE%', $CatalinaBase
            $ClassPath = $ClassPath -replace '%JAVA_HOME%', $JavaHome
            $ClassPath = $ClassPath -replace '%JRE_HOME%', $JavaHome

            if ( $ClassPath )
            {
                Write-Verbose "Class path string:`n$ClassPath`n`n"
            }
           

            #
            # Set defaults if values not specified.
            #
            if ( $Action -eq "INSTALL" -and -not $Display )
            {
                $Display = $DEFAULT_DISPLAY_VALUE
            }

            if ( $Action -eq "INSTALL" -and -not $Description )
            {
                $Description = $DEFAULT_DESCRIPTION_VALUE
            }


            Write-Debug "Display = $Display"
            Write-Debug "Description = $Description"

            [String] $CommandArgs = "--Install `"$TomcatExe`" " +
                                    "--DisplayName `"$Display`" " +
                                    "--Description `"$Description`" " +
                                    "--LogLevel WARN " +
                                    "--LogPath `"$CatalinaBase\logs`" " +
                                    "--PidFile tcserver.pid " +
                                    "--Environment `"$EnvironmentVars`" " +
                                    "--JavaHome `"$JavaHomeDir`" " +
                                    "--Jvm `"$JavaHomeDir\bin\server\jvm.dll`" " +
                                    "--StartMode jvm " +
                                    "--StopMode jvm " +
                                    "--Startup auto " +
                                    "--StartPath `"$CatalinaHome`" " +
                                    "--StopPath `"$CatalinaHome`" " +
                                    "--StartClass org.apache.catalina.startup.Bootstrap " +
                                    "--StopClass org.apache.catalina.startup.Bootstrap " +
                                    "--StartParams start " +
                                    "--StopParams stop " +
                                    "--Classpath `'$ClassPath`' " +
                                    "--JvmOptions `"$JVMOptionList`" " +
                                    "--JvmOptions9 `"--add-opens=java.base/java.lang=ALL-UNNAMED#--add-opens=java.base/java.io=ALL-UNNAMED#--add-opens=java.rmi/sun.rmi.transport=ALL-UNNAMED`" "  +
                                    "$JvmMs_Option" +
                                    "$JmvMx_Option" +
                                    "--StdOutput `"$CatalinaBase\logs\wrapper.log`" " +
                                    "--StdError `"$CatalinaBase\logs\wrapper.log`" "

            $Command = "& `"$TomcatExe`" `"//IS//$Name`" $CommandArgs"
            $SC_Command = "& sc.exe config `"$Name`" obj= `"LocalSystem`""
            if ( $NoExecMode -eq $true )
            {
                Write-Output "Simulated execution of:"
                ( $Command -split " --" ) -join "`r`n--"
                $SC_Command
            }

            else
            {
                Invoke-Expression -command $Command
                $Status = $LastExitCode
                if ( $Status -ne 0 )
                {
                    Write-Output "INSTALL returned code $Status"
                    Write-Error "ERROR: Failed to install the '$Name' service"
                }

                else
                {
                    Write-Output "NOTE: The '$Name' service has been installed."
                    Invoke-Expression -command $SC_Command
                }
            }
        }

        "UNINSTALL"
        {
            $Command = "& `"$TomcatExe`" `"//DS//$Name`""
            if ( $NoExecMode -eq $true )
            {
                Write-Output "Simulated execution of: $Command"
            }

            else
            {
                Invoke-Expression -command $Command
                $Status = $LastExitCode
                if ( $Status -ne 0 )
                {
                    Write-Output "UNINSTALL returned code $Status"
                    Write-Error "ERROR: Failed to uninstall the '$Name' service"
                }

                else
                {
                    Write-Output "NOTE: The '$Name' service has been uninstalled."
                }
            }
        }

        "START"
        {
            $Command = "Start-Service -Name `"$Name`" -ErrorAction `"Continue`" -ErrorVariable ErrorStatus"
            if ( $NoExecMode -eq $true )
            {
                Write-Output "Simulated execution of: $Command"
            }

            else
            {
                Invoke-Expression -command $Command

                if ( $ErrorStatus.Count -ne 0 )
                {
                    Write-Error "ERROR: Failed to start the '$Name' service"
                }

                else
                {
                    Write-Output "NOTE: The '$Name' service has been started."
                }
            }
        }

        "STOP"
        {
            $Command = "Stop-Service -Name `"$Name`" -ErrorAction `"Continue`" -ErrorVariable ErrorStatus"
            if ( $NoExecMode -eq $true )
            {
                Write-Output "Simulated execution of: $Command"
            }

            else
            {
                Invoke-Expression -command $Command

                if ( $ErrorStatus.Count -ne 0 )
                {
                    Write-Error "ERROR: Failed to stop the '$Name' service"
                }

                else
                {
                    Write-Output "NOTE: The '$Name' service has been stopped."
                }
            }
        }

        # Should never happen
        Default { Write-Error "Invalid action specified: $Action" }
    }

    $DebugPreference = $InitialDebugPreference
    $VerbosePreference = $InitialVerbosePreference
    $ErrorActionPreference = $InitialErrorActionPreference

    exit $Status
}

#
# Invoke the main funciton for the script.
#
Main
