/*
 * Decompiled with CFR 0.152.
 */
package com.atomikos.recovery.fs;

import com.atomikos.icatch.config.Configuration;
import com.atomikos.icatch.provider.ConfigProperties;
import com.atomikos.logging.Logger;
import com.atomikos.logging.LoggerFactory;
import com.atomikos.persistence.imp.LogFileLock;
import com.atomikos.recovery.LogException;
import com.atomikos.recovery.LogReadException;
import com.atomikos.recovery.LogWriteException;
import com.atomikos.recovery.PendingTransactionRecord;
import com.atomikos.recovery.fs.Repository;
import com.atomikos.util.VersionedFile;
import java.io.BufferedReader;
import java.io.EOFException;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.ObjectStreamException;
import java.io.StreamCorruptedException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

public class FileSystemRepository
implements Repository {
    private static final Logger LOGGER = LoggerFactory.createLogger(FileSystemRepository.class);
    private static int EXPECTED_LOG_RECORD_SIZE_IN_BYTES = 100;
    private VersionedFile file;
    private FileChannel rwChannel = null;
    private LogFileLock lock_;
    private long minUsableSpaceRequired;

    @Override
    public void init() throws LogException {
        ConfigProperties configProperties = Configuration.getConfigProperties();
        String baseDir = configProperties.getLogBaseDir();
        String baseName = configProperties.getLogBaseName();
        LOGGER.logDebug("baseDir " + baseDir);
        LOGGER.logDebug("baseName " + baseName);
        this.lock_ = new LogFileLock(baseDir, baseName);
        LOGGER.logDebug("LogFileLock " + this.lock_);
        this.lock_.acquireLock();
        this.file = new VersionedFile(baseDir, baseName, ".log");
        this.minUsableSpaceRequired = configProperties.getCheckpointInterval() * (long)EXPECTED_LOG_RECORD_SIZE_IN_BYTES * 2L;
    }

    @Override
    public void put(String id, PendingTransactionRecord pendingTransactionRecord) throws IllegalArgumentException, LogWriteException {
        try {
            this.initChannelIfNecessary();
            this.write(pendingTransactionRecord, true);
        }
        catch (IOException e) {
            throw new LogWriteException((Throwable)e);
        }
    }

    private synchronized void initChannelIfNecessary() throws FileNotFoundException {
        if (this.rwChannel == null) {
            this.rwChannel = this.file.openNewVersionForNioWriting();
        }
    }

    private void write(PendingTransactionRecord pendingTransactionRecord, boolean flushImmediately) throws IOException {
        String str = pendingTransactionRecord.toRecord();
        byte[] buffer = str.getBytes();
        ByteBuffer buff = ByteBuffer.wrap(buffer);
        this.writeToFile(buff, !pendingTransactionRecord.state.isFinalState() && flushImmediately);
    }

    private void writeToFile(ByteBuffer buff, boolean force) throws IOException {
        this.rwChannel.write(buff);
        if (force) {
            this.rwChannel.force(false);
        }
    }

    @Override
    public PendingTransactionRecord get(String coordinatorId) throws LogReadException {
        throw new UnsupportedOperationException();
    }

    @Override
    public Collection<PendingTransactionRecord> findAllCommittingCoordinatorLogEntries() throws LogReadException {
        throw new UnsupportedOperationException();
    }

    @Override
    public Collection<PendingTransactionRecord> getAllCoordinatorLogEntries() throws LogReadException {
        FileInputStream fis = null;
        try {
            fis = this.file.openLastValidVersionForReading();
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
        if (fis != null) {
            return FileSystemRepository.readFromInputStream(fis);
        }
        return Collections.emptyList();
    }

    public static Collection<PendingTransactionRecord> readFromInputStream(InputStream in) throws LogReadException {
        Map<Object, Object> coordinatorLogEntries = new HashMap();
        BufferedReader br = null;
        try {
            InputStreamReader isr = new InputStreamReader(in);
            br = new BufferedReader(isr);
            coordinatorLogEntries = FileSystemRepository.readContent(br);
        }
        catch (Exception e) {
            try {
                LOGGER.logFatal("Error in recover", (Throwable)e);
                throw new LogReadException((Throwable)e);
            }
            catch (Throwable throwable) {
                FileSystemRepository.closeSilently(br);
                throw throwable;
            }
        }
        FileSystemRepository.closeSilently(br);
        return coordinatorLogEntries.values();
    }

    static Map<String, PendingTransactionRecord> readContent(BufferedReader br) throws IOException {
        HashMap<String, PendingTransactionRecord> coordinatorLogEntries = new HashMap<String, PendingTransactionRecord>();
        String line = null;
        try {
            while ((line = br.readLine()) != null) {
                if (line.startsWith("{\"id\"")) {
                    String msg = "Detected old log file format - please terminate all transactions under the old release first!";
                    throw new IOException(msg);
                }
                PendingTransactionRecord coordinatorLogEntry = PendingTransactionRecord.fromRecord((String)line);
                coordinatorLogEntries.put(coordinatorLogEntry.id, coordinatorLogEntry);
            }
        }
        catch (EOFException unexpectedEOF) {
            LOGGER.logTrace("Unexpected EOF - logfile not closed properly last time?", (Throwable)unexpectedEOF);
        }
        catch (StreamCorruptedException unexpectedEOF) {
            LOGGER.logTrace("Unexpected EOF - logfile not closed properly last time?", (Throwable)unexpectedEOF);
        }
        catch (ObjectStreamException unexpectedEOF) {
            LOGGER.logTrace("Unexpected EOF - logfile not closed properly last time?", (Throwable)unexpectedEOF);
        }
        catch (IllegalArgumentException couldNotParseLastRecord) {
            LOGGER.logTrace("Unexpected record format - logfile not closed properly last time?", (Throwable)couldNotParseLastRecord);
        }
        catch (RuntimeException unexpectedEOF) {
            LOGGER.logWarning("Unexpected EOF - logfile not closed properly last time? " + line + " " + unexpectedEOF);
        }
        return coordinatorLogEntries;
    }

    private static void closeSilently(BufferedReader fis) {
        try {
            if (fis != null) {
                fis.close();
            }
        }
        catch (IOException io) {
            LOGGER.logWarning("Fail to close logfile after reading - ignoring");
        }
    }

    @Override
    public void writeCheckpoint(Collection<PendingTransactionRecord> checkpointContent) throws LogWriteException {
        try {
            this.closeOutput();
            this.rwChannel = this.file.openNewVersionForNioWriting();
            for (PendingTransactionRecord coordinatorLogEntry : checkpointContent) {
                this.write(coordinatorLogEntry, false);
            }
            this.rwChannel.force(false);
            this.file.discardBackupVersion();
        }
        catch (FileNotFoundException fileNotFoundException) {
        }
        catch (Exception e) {
            LOGGER.logFatal("Failed to write checkpoint", (Throwable)e);
            throw new LogWriteException((Throwable)e);
        }
        this.assertSufficientDiskSpaceLeft();
    }

    private void assertSufficientDiskSpaceLeft() {
        long availableBytesOnDisk = this.file.getUsableSpace();
        if (availableBytesOnDisk < this.minUsableSpaceRequired * 2L) {
            LOGGER.logWarning("Disk almost full - please ensure sufficient free space for the transaction logs!");
        } else if (availableBytesOnDisk < this.minUsableSpaceRequired) {
            LOGGER.logFatal("Disk almost full - please ensure sufficient free space for the transaction logs!");
        }
    }

    protected void closeOutput() throws IllegalStateException {
        try {
            if (this.file != null) {
                this.file.close();
            }
        }
        catch (IOException e) {
            throw new IllegalStateException("Error closing previous output", e);
        }
    }

    @Override
    public void close() {
        try {
            this.closeOutput();
        }
        catch (Exception e) {
            LOGGER.logWarning("Error closing file - ignoring", (Throwable)e);
        }
        finally {
            this.lock_.releaseLock();
        }
    }
}

