/*
 * Decompiled with CFR 0.152.
 */
package org.apache.geode.deployment.internal;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.geode.annotations.internal.MakeNotStatic;
import org.apache.geode.deployment.internal.DeployedJar;
import org.apache.geode.internal.classloader.ClassPathLoader;
import org.apache.geode.logging.internal.log4j.api.LogService;
import org.apache.geode.management.internal.utils.JarFileUtils;
import org.apache.logging.log4j.Logger;

public class JarDeployer
implements Serializable {
    private static final long serialVersionUID = 1L;
    private static final Logger logger = LogService.getLogger();
    private static final Pattern POUND_VERSION_SCHEME = Pattern.compile("^vf\\.gf#(?<artifact>.*)\\.jar#(?<version>\\d+)$");
    @MakeNotStatic
    private static final Lock lock = new ReentrantLock();
    private final Map<String, DeployedJar> deployedJars = new ConcurrentHashMap<String, DeployedJar>();
    private final File deployDirectory;

    public JarDeployer() {
        this(new File(System.getProperty("user.dir")));
    }

    public JarDeployer(File deployDirectory) {
        this.deployDirectory = deployDirectory == null ? new File(System.getProperty("user.dir")) : deployDirectory;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DeployedJar deployWithoutRegistering(String artifactId, File stagedJar) throws IOException {
        lock.lock();
        try {
            boolean shouldDeployNewVersion = this.shouldDeployNewVersion(artifactId, stagedJar);
            if (!shouldDeployNewVersion) {
                logger.debug("No need to deploy a new version of {}", (Object)stagedJar.getName());
                DeployedJar deployedJar = null;
                return deployedJar;
            }
            this.verifyWritableDeployDirectory();
            Path deployedFile = this.getNextVersionedJarFile(stagedJar.getName()).toPath();
            FileUtils.copyFile((File)stagedJar, (File)deployedFile.toFile());
            DeployedJar deployedJar = new DeployedJar(deployedFile.toFile());
            return deployedJar;
        }
        finally {
            lock.unlock();
        }
    }

    protected File getNextVersionedJarFile(String unversionedJarName) {
        int maxVersion = this.getMaxVersion(JarFileUtils.getArtifactId(unversionedJarName));
        String nextVersionJarName = FilenameUtils.getBaseName((String)unversionedJarName) + ".v" + (maxVersion + 1) + ".jar";
        logger.debug("Next versioned jar name for {} is {}", (Object)unversionedJarName, (Object)nextVersionJarName);
        return new File(this.deployDirectory, nextVersionJarName);
    }

    protected int getMaxVersion(String artifactId) {
        return Arrays.stream(this.deployDirectory.list()).filter(x -> artifactId.equals(JarFileUtils.toArtifactId(x))).map(JarFileUtils::extractVersionFromFilename).reduce(Integer::max).orElse(0);
    }

    public void verifyWritableDeployDirectory() throws IOException {
        try {
            if (this.deployDirectory.canWrite()) {
                return;
            }
        }
        catch (SecurityException ex) {
            throw new IOException("Unable to write to deploy directory", ex);
        }
        throw new IOException("Unable to write to deploy directory: " + this.deployDirectory.getCanonicalPath());
    }

    protected void renameJarsWithOldNamingConvention() throws IOException {
        Set<File> jarsWithOldNamingConvention = this.findJarsWithOldNamingConvention();
        if (jarsWithOldNamingConvention.isEmpty()) {
            return;
        }
        for (File jar : jarsWithOldNamingConvention) {
            this.renameJarWithOldNamingConvention(jar);
        }
    }

    protected Set<File> findJarsWithOldNamingConvention() {
        return Stream.of(this.deployDirectory.listFiles()).filter(file -> this.isOldNamingConvention(file.getName())).collect(Collectors.toSet());
    }

    protected boolean isOldNamingConvention(String fileName) {
        return POUND_VERSION_SCHEME.matcher(fileName).matches();
    }

    private void renameJarWithOldNamingConvention(File oldJar) throws IOException {
        Matcher matcher = POUND_VERSION_SCHEME.matcher(oldJar.getName());
        if (!matcher.matches()) {
            throw new IllegalArgumentException("The given jar " + oldJar.getCanonicalPath() + " does not match the old naming convention");
        }
        String unversionedJarNameWithoutExtension = matcher.group(1);
        String jarVersion = matcher.group(2);
        String newJarName = unversionedJarNameWithoutExtension + ".v" + jarVersion + ".jar";
        File newJar = new File(this.deployDirectory, newJarName);
        logger.debug("Renaming deployed jar from {} to {}", (Object)oldJar.getCanonicalPath(), (Object)newJar.getCanonicalPath());
        FileUtils.moveFile((File)oldJar, (File)newJar);
    }

    public Map<String, DeployedJar> getLatestVersionOfJarsOnDisk() {
        logger.info("Loading previously deployed jars");
        lock.lock();
        try {
            this.verifyWritableDeployDirectory();
            this.renameJarsWithOldNamingConvention();
            Map<String, Integer> artifactToMaxVersion = this.findArtifactsAndMaxVersion();
            HashMap<String, DeployedJar> latestVersionOfEachJar = new HashMap<String, DeployedJar>();
            for (File file : this.deployDirectory.listFiles()) {
                String artifactId = JarFileUtils.toArtifactId(file.getName());
                if (artifactId == null) continue;
                int version = JarFileUtils.extractVersionFromFilename(file.getName());
                if (version < artifactToMaxVersion.get(artifactId)) {
                    FileUtils.deleteQuietly((File)file);
                    continue;
                }
                DeployedJar deployedJar = new DeployedJar(file);
                latestVersionOfEachJar.put(deployedJar.getArtifactId(), deployedJar);
            }
            HashMap<String, DeployedJar> hashMap = latestVersionOfEachJar;
            return hashMap;
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        finally {
            lock.unlock();
        }
    }

    Map<String, Integer> findArtifactsAndMaxVersion() {
        HashMap<String, Integer> artifactToMaxVersion = new HashMap<String, Integer>();
        for (String fileName : this.deployDirectory.list()) {
            String artifactId = JarFileUtils.toArtifactId(fileName);
            if (artifactId == null) continue;
            int version = JarFileUtils.extractVersionFromFilename(fileName);
            Integer maxVersion = (Integer)artifactToMaxVersion.get(artifactId);
            if (maxVersion != null && maxVersion >= version) continue;
            artifactToMaxVersion.put(artifactId, version);
        }
        return artifactToMaxVersion;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void registerNewVersions(String artifactId, DeployedJar deployedJar) throws ClassNotFoundException {
        lock.lock();
        try {
            if (deployedJar != null) {
                logger.info("Registering new version of jar: {}", (Object)deployedJar);
                DeployedJar oldJar = this.deployedJars.put(artifactId, deployedJar);
                ClassPathLoader.getLatest().chainClassloader(deployedJar.getFile());
            }
        }
        finally {
            lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DeployedJar deploy(File stagedJarFile) throws IOException, ClassNotFoundException {
        if (!JarFileUtils.hasValidJarContent(stagedJarFile)) {
            throw new IllegalArgumentException("File does not contain valid JAR content: " + stagedJarFile.getName());
        }
        lock.lock();
        try {
            String artifactId = JarFileUtils.getArtifactId(stagedJarFile.getName());
            DeployedJar deployedJar = this.deployWithoutRegistering(artifactId, stagedJarFile);
            this.registerNewVersions(artifactId, deployedJar);
            DeployedJar deployedJar2 = deployedJar;
            return deployedJar2;
        }
        finally {
            lock.unlock();
        }
    }

    private boolean shouldDeployNewVersion(String artifactId, File stagedJar) throws IOException {
        DeployedJar oldDeployedJar = this.deployedJars.get(artifactId);
        if (oldDeployedJar == null) {
            return true;
        }
        if (oldDeployedJar.hasSameContentAs(stagedJar)) {
            logger.warn("Jar is identical to the latest deployed version: {}", (Object)oldDeployedJar.getFileCanonicalPath());
            return false;
        }
        return true;
    }

    public Map<String, DeployedJar> getDeployedJars() {
        return Collections.unmodifiableMap(this.deployedJars);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String undeploy(String artifactId) throws IOException {
        lock.lock();
        logger.debug("JarDeployer Undeploying artifactId: {}", (Object)artifactId);
        try {
            logger.debug("JarDeployer deployedJars list before remove: {}", (Object)Arrays.toString(this.deployedJars.keySet().toArray()));
            DeployedJar deployedJar = this.deployedJars.remove(artifactId);
            if (deployedJar == null) {
                throw new IllegalArgumentException(artifactId + " not deployed");
            }
            logger.debug("JarDeployer deployedJars list after remove: {}", (Object)Arrays.toString(this.deployedJars.keySet().toArray()));
            ClassPathLoader.getLatest().unloadClassloaderForArtifact(artifactId);
            this.deleteAllVersionsOfJar(deployedJar.getFile().getName());
            String string = deployedJar.getFileCanonicalPath();
            return string;
        }
        finally {
            lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteAllVersionsOfJar(String jarName) {
        lock.lock();
        logger.info("Deleting all versions of jar: {}", (Object)jarName);
        String artifactId = JarFileUtils.toArtifactId(jarName);
        if (artifactId == null) {
            artifactId = JarFileUtils.getArtifactId(jarName);
        }
        logger.debug("ArtifactId to delete: {}", (Object)artifactId);
        try {
            for (File file : this.deployDirectory.listFiles()) {
                logger.debug("File in deploy directory: {} with artifactId: {}", (Object)file.getName(), (Object)JarFileUtils.toArtifactId(file.getName()));
                if (!artifactId.equals(JarFileUtils.toArtifactId(file.getName()))) continue;
                logger.info("Deleting: {}", (Object)file.getAbsolutePath());
                FileUtils.deleteQuietly((File)file);
            }
        }
        finally {
            lock.unlock();
        }
    }

    public String toString() {
        return this.getClass().getName() + '@' + System.identityHashCode(this) + '{' + "deployDirectory=" + this.deployDirectory + '}';
    }
}

